/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "blinds-window.h"

#include <gdk/gdkx.h>
#include <libwnck/libwnck.h>

#include <X11/extensions/Xfixes.h>
#include <X11/extensions/shape.h>

G_DEFINE_TYPE (BlindsWindow, blinds_window, GTK_TYPE_WINDOW);

#define BLINDS_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  BLINDS_TYPE_WINDOW, \
  BlindsWindowPrivate))

struct _BlindsWindowPrivate
{
  WnckScreen *screen;

  gint opacity;
  gint tag;
};

/*
 * Window stacking stuff 
 */

/*
 * We need the window-frame's Window, as that is a sibling of our window!
 */
static Window
frame_window (WnckWindow *window)
{
  Window   active;
  Display *display = GDK_DISPLAY ();
  Window   root_return;
  Window   parent_return;
  Window  *children_return;
  guint    nchildren_return;
  
  active = wnck_window_get_xid (window);

  gdk_error_trap_push ();
  XQueryTree (display, active, &root_return, &parent_return, 
              &children_return, &nchildren_return);
  gdk_flush ();
  gdk_error_trap_pop ();

  if (children_return)
    XFree (children_return);

  return parent_return;
}

static void
on_active_window_changed (WnckScreen   *screen,
                          WnckWindow   *old_window,
                          BlindsWindow *blinds_window)
{
  BlindsWindowPrivate *priv;
  GtkWidget *widget;
  WnckWindow *active_window;
  XWindowChanges changes;
  gint res;
  
  g_return_if_fail (BLINDS_IS_WINDOW (blinds_window));
  priv = blinds_window->priv;

  widget = GTK_WIDGET (blinds_window);
  active_window = wnck_screen_get_active_window (screen);

  if (!WNCK_IS_WINDOW (active_window))
  {
    g_debug ("Received a bad window from wnck");
    return;
  }

  if (wnck_window_get_xid (active_window) == GDK_DRAWABLE_XID (widget->window))
  {
    g_debug ("Blinds window is active, woops!");
    return;
  }
  g_debug ("Active window: %s", wnck_window_get_name (active_window));

  /* Change the stacking order so blinds is beneath the active window :-/ */
  changes.sibling = frame_window (active_window);
  changes.stack_mode = Below;
    
  gdk_error_trap_push ();
  XConfigureWindow (GDK_DISPLAY(),
                    GDK_WINDOW_XID (widget->window),
                    CWSibling | CWStackMode, &changes);
  gdk_flush ();
  if ((res = gdk_error_trap_pop ()))
  {
    g_warning ("Received X error: %d\n", res);       
  }    
}

/*
 * Allow events to passthrough the blinds window
 */
static void
allow_input_passthrough (GtkWidget *widget)
{
  Display      *dpy;
  Window        w;
  XserverRegion region;

  gtk_widget_show (widget);
  
  dpy = GDK_DRAWABLE_XDISPLAY (widget->window);
  w = GDK_DRAWABLE_XID (widget->window);
  
  region = XFixesCreateRegion (dpy, NULL, 0);

  gdk_error_trap_push ();

  XFixesSetWindowShapeRegion (dpy, w, ShapeBounding, 0, 0, 0);
  XFixesSetWindowShapeRegion (dpy, w, ShapeInput, 0, 0, region);

  gdk_flush ();
  gdk_error_trap_pop ();

  XFixesDestroyRegion (dpy, region);
}

/*
 * Drawing operations
 */
static gboolean
on_expose (GtkWidget *widget, GdkEventExpose *event)
{
  GdkRegion       *region;
  cairo_t         *cr;
  cairo_surface_t *image;
  cairo_pattern_t *pattern;
  gint             width, height;
  
  cr = gdk_cairo_create (widget->window);
  if (!cr)
    return FALSE;

  width = widget->allocation.width;
  height = widget->allocation.height;

  /* Clip */
  region = gdk_region_rectangle (&widget->allocation);
  gdk_cairo_region (cr, event->region);
  cairo_clip (cr);
  gdk_region_destroy (region);
 
  cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
  cairo_paint (cr);

  cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

  cairo_set_source_rgba (cr, 0.0f, 0.0f, 0.0f, 0.2f);
  cairo_rectangle (cr, 0, 0, width, height);
  cairo_fill (cr);

  /* Tile pattern to make the inactive windows seem more obscure */
  image = cairo_image_surface_create_from_png (PKGDATADIR"/tile.png");
 
  pattern = cairo_pattern_create_for_surface (image);
  cairo_pattern_set_extend (pattern, CAIRO_EXTEND_REPEAT);

  cairo_set_source (cr, pattern);
  cairo_rectangle (cr, 0, 0, width, height);
  cairo_fill (cr);

  cairo_pattern_destroy (pattern);
  cairo_surface_destroy (image);    

  /* Some more gradients */
  pattern = cairo_pattern_create_linear (0, 0, width, height);
  cairo_pattern_add_color_stop_rgba (pattern, 0.0, 1.0f, 1.0f, 1.0f, 0.5f);
  cairo_pattern_add_color_stop_rgba (pattern, 0.25, 1.0f, 1.0f, 1.0f, 0.1f);
  cairo_pattern_add_color_stop_rgba (pattern, 0.5, 1.0f, 1.0f, 1.0f, 0.4f);
  cairo_pattern_add_color_stop_rgba (pattern, 0.75, 1.0f, 1.0f, 1.0f, 0.1f);
  cairo_pattern_add_color_stop_rgba (pattern, 1.0, 1.0f, 1.0f, 1.0f, 0.5f);

  cairo_set_source (cr, pattern);
  cairo_rectangle (cr, 0, 0, width, height);
  cairo_fill (cr);

  cairo_pattern_destroy (pattern);

  pattern = cairo_pattern_create_linear (0, height, width, 0);
  cairo_pattern_add_color_stop_rgba (pattern, 1.0, 0.0f, 0.0f, 0.0f, 0.3f);
  cairo_pattern_add_color_stop_rgba (pattern, 0.75, 0.0f, 0.0f, 0.0f, 0.1f);
  cairo_pattern_add_color_stop_rgba (pattern, 0.5, 0.0f, 0.0f, 0.0f, 0.3f);
  cairo_pattern_add_color_stop_rgba (pattern, 0.25, 0.0f, 0.0f, 0.0f, 0.1f);
  cairo_pattern_add_color_stop_rgba (pattern, 0.0, 0.0f, 0.0f, 0.0f, 0.3f);

  cairo_set_source (cr, pattern);
  cairo_rectangle (cr, 0, 0, width, height);
  cairo_fill (cr);

  cairo_pattern_destroy (pattern);

  cairo_destroy (cr);

  return TRUE;
}

/* GObject stuff */
static void
blinds_window_finalize (GObject *object)
{
  BlindsWindowPrivate *priv;

  priv = BLINDS_WINDOW_GET_PRIVATE (object);

  G_OBJECT_CLASS (blinds_window_parent_class)->finalize (object);
}

static void
blinds_window_class_init (BlindsWindowClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass      *wid_class = GTK_WIDGET_CLASS (klass);

  obj_class->finalize = blinds_window_finalize;

  wid_class->expose_event = on_expose;

  g_type_class_add_private (obj_class, sizeof (BlindsWindowPrivate));
}

static void
blinds_window_init (BlindsWindow *window)
{
  BlindsWindowPrivate *priv;
  GdkScreen *screen;
  GdkColormap *map;

  priv = window->priv = BLINDS_WINDOW_GET_PRIVATE (window);

  gtk_widget_set_app_paintable (GTK_WIDGET (window), TRUE);

  screen = gtk_widget_get_screen (GTK_WIDGET (window));
  map = gdk_screen_get_rgba_colormap (screen);

  if (!map)
  {
    g_warning ("Unable to run in a non-composited environment");
    gtk_main_quit ();
  }

  gtk_widget_set_colormap (GTK_WIDGET (window), map);
  gtk_window_maximize (GTK_WINDOW (window));
  
  g_signal_connect_after (window, "show", 
                          G_CALLBACK (allow_input_passthrough), NULL);

  /* We need to be notified when the active window changes on the desktop */
  priv->screen = wnck_screen_get_default ();
  g_signal_connect (priv->screen, "active-window-changed",
                    G_CALLBACK (on_active_window_changed), window);

  gtk_window_resize (GTK_WINDOW (window),
                     gdk_screen_get_width (screen),
                     gdk_screen_get_height (screen));
}

GtkWidget *
blinds_window_new (void)
{
  GtkWidget *window = NULL;

  window = g_object_new (BLINDS_TYPE_WINDOW,
                         "type", GTK_WINDOW_POPUP,
                         "type-hint", GDK_WINDOW_TYPE_HINT_NORMAL,
                         "skip-pager-hint", TRUE,
                         "skip-taskbar-hint", TRUE,
                         "decorated", FALSE,
                         "focus-on-map", FALSE,
                         "border-width", 0, 
                         NULL);

  return window;
}
