------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2004 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-environments.adb,v 1.7 2004/06/28 04:57:05 vagul Exp $

with Ada.Text_IO;

with System;

package body OCI.Environments is

   use type SWord, Ub4, OCIHandle;

   Create_Mode : Ub4 := OCI_THREADED;

   function Alloc_Memory
     (ctxp : in System.Address;
      Size : in Integer)
      return System.Address;
   pragma Convention (C, Alloc_Memory);

   function Realloc_Memory
     (ctxp   : System.Address;
      Memptr : System.Address;
      Size   : Integer)
      return System.Address;
   pragma Convention (C, Realloc_Memory);

   procedure Free_Memory
     (ctxp   : in System.Address;
      Memptr : in System.Address);
   pragma Convention (C, Free_Memory);

   function Malloc (Size : in Integer) return System.Address;
   pragma Import (C, Malloc, "__gnat_malloc");

   procedure Free (Address : in System.Address);
   pragma Import (C, Free, "__gnat_free");

   function Realloc
     (Ptr    : in System.Address;
      Size   : in Integer)
      return System.Address;
   pragma Import (C, Realloc, "__gnat_realloc");

   ------------------
   -- Alloc_Memory --
   ------------------

   function Alloc_Memory
     (ctxp : in System.Address;
      Size : in Integer)
      return System.Address
   is
      pragma Unreferenced (ctxp);
   begin
      Counter.Add (1);
      Last_Size := Size;
      return Malloc (Size);
   end Alloc_Memory;

   -------------
   -- Counter --
   -------------

   protected body Counter is

      ---------
      -- Add --
      ---------

      procedure Add (It : Integer) is
      begin
         Val := Val + It;
      end Add;

      -----------
      -- Value --
      -----------

      function Value return Integer is
      begin
         return Val;
      end Value;

   end Counter;

   ------------
   -- Create --
   ------------

   function Create return Thread_Environment is
      Env         : aliased OCIEnv := OCIEnv (Empty_Handle);
      Rc          : SWord;
      Environment : Thread_Environment;
   begin
      Rc := OCIEnvCreate
        (Env'Access,
         Create_Mode,
         Malocfp  => Alloc_Memory'Address,
         Ralocfp  => Realloc_Memory'Address,
         Mfreefp  => Free_Memory'Address);

      if Rc /= OCI_SUCCESS then
         raise Environment_Creation_Error;
      end if;

      Environment.Handle := Env;
      return Environment;
   end Create;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Object : in out Thread_Environment) is
      Rc : SWord;
      use Ada.Text_IO;
   begin
      if Object.Handle /= OCIEnv (Empty_Handle) then
         Rc := OCIHandleFree (OCIHandle (Object.Handle), OCI_HTYPE_ENV);

         Object.Handle := OCIEnv (Empty_Handle);

         if Rc /= OCI_SUCCESS then
            Put_Line
              (Current_Error,
               "Warning: Return code on free environment "
                  & SWord'Image (Rc));
         end if;
      end if;
   end Destroy;

   -----------------
   -- Free_Memory --
   -----------------

   procedure Free_Memory
     (ctxp   : in System.Address;
      Memptr : in System.Address)
   is
      pragma Unreferenced (ctxp);
   begin
      Counter.Add (-1);
      Free (Memptr);
   end Free_Memory;

   --------------------
   -- Realloc_Memory --
   --------------------

   function Realloc_Memory
     (ctxp   : in System.Address;
      Memptr : in System.Address;
      Size   : in Integer)
      return System.Address
   is
      pragma Unreferenced (ctxp);
   begin
      if Size = 0 then
         Counter.Add (-1);
      end if;

      return Realloc (Memptr, Size);
   end Realloc_Memory;

   --------------------------
   -- Set_Create_Mode_Flag --
   --------------------------

   procedure Set_Create_Mode_Flag (Flag : in Ub4) is
   begin
      Create_Mode := Create_Mode or Flag;
   end Set_Create_Mode_Flag;

end OCI.Environments;
