/*
 * Copyright (C) 2010 Freescale Semiconductor, Inc. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_isink.c
 *
 * Description:    Implementation of Video Sink Plugin based on FSL IPU LIB API
 *                 for Gstreamer
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 */

/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <gst/gst.h>
#include <gst/video/gstvideosink.h>
#include <linux/videodev.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdint.h>
#include <sys/mman.h>
#include <string.h>
#include "mfw_gst_utils.h"
#include <stdio.h>
#include <unistd.h>
#include "linux/mxcfb.h"

#include "video_surface.h"
#include <sys/time.h>

#if USE_X11
#include "mfw_gst_i_xlib.h"
#endif

#include "mfw_gst_isink.h"
#include "buffer_allocator.h"

/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/
/* None */

/*=============================================================================
                LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
=============================================================================*/


typedef enum {
    ISINK_PROP_0,
    ISINK_PROP_ROTATION_0,
    ISINK_PROP_DISPWIN_LEFT_0,
    ISINK_PROP_DISPWIN_TOP_0,
    ISINK_PROP_DISPWIN_WIDTH_0,
    ISINK_PROP_DISPWIN_HEIGHT_0,
    ISINK_PROP_INPUT_CROP_LEFT_0,
    ISINK_PROP_INPUT_CROP_RIGHT_0,
    ISINK_PROP_INPUT_CROP_TOP_0,
    ISINK_PROP_INPUT_CROP_BOTTOM_0,
    ISINK_PROP_SETPARA
}ISINK_PROPERTY_T;

typedef enum{
    RGB_SPACE,
    YUV_SPACE
}ColorSpace;

typedef struct {
    guint32 fourcc;
    guint32 ifmt;
    gint bpp;
}ISinkFmt;

/*=============================================================================
                              LOCAL MACROS
=============================================================================*/
/* used for debugging */
#define ISINK_ERROR(format,...)  g_print(RED_STR(format, ##__VA_ARGS__))
#define ISINK_FLOW(format,...)  g_print(GREEN_STR(format, ##__VA_ARGS__))
#define ISINK_FLOW_DEFAULT ISINK_FLOW("%s:%d\n", __FUNCTION__, __LINE__)

#define ISINK_MALLOC g_malloc
#define ISINK_FREE g_free

#define ISINK_MEMCPY memcpy

#define LCD_FB_NUM 2
#define NTSC 0
#define PAL 1

#define DEFAULT_LCD_ROTATION 4
#define DEFAULT_TV_ROTATION 0


#define COLORKEY_RED        1
#define COLORKEY_GREEN      2
#define COLORKEY_BLUE       3
    
#define RGB888(r,g,b)\
        ((((guint32)(r))<<16)|(((guint32)(g))<<8)|(((guint32)(b))))
#define RGB888TORGB565(rgb)\
        ((((rgb)<<8)>>27<<11)|(((rgb)<<18)>>26<<5)|(((rgb)<<27)>>27))
    
#define RGB565TOCOLORKEY(rgb)                              \
          ( ((rgb & 0xf800)<<8)  |  ((rgb & 0xe000)<<3)  |     \
            ((rgb & 0x07e0)<<5)  |  ((rgb & 0x0600)>>1)  |     \
            ((rgb & 0x001f)<<3)  |  ((rgb & 0x001c)>>2)  )
            
#define FOURCC_NONE 0x0

/*=============================================================================
                             STATIC VARIABLES
=============================================================================*/

static GstElementDetails mfw_gst_isink_details =
GST_ELEMENT_DETAILS("Freescale: i_sink",
		    "Sink/Video",
		    "Video rendering plugin support DR",
		    FSL_GST_MM_PLUGIN_AUTHOR);



static ISinkFmt isink_yuv_fmts[] = 	{
    {GST_MAKE_FOURCC('I', '4', '2', '0'),GST_MAKE_FOURCC('I', '4', '2', '0'), 12}, 
    {GST_MAKE_FOURCC('N', 'V', '1', '2'),GST_MAKE_FOURCC('N', 'V', '1', '2'), 12},
    {GST_MAKE_FOURCC('Y', 'U', 'Y', '2'),GST_MAKE_FOURCC('Y', 'U', 'Y', 'V'), 16},
    {FOURCC_NONE, 0}
};

static ISinkFmt isink_rgb_fmts[] = {
    {GST_MAKE_FOURCC('R', 'G', 'B', 'P'), GST_MAKE_FOURCC('R', 'G', 'B', 'P'), 16},
    {GST_MAKE_FOURCC('R', 'G', 'B', '3'), GST_MAKE_FOURCC('R', 'G', 'B', '3'), 24},
    {GST_MAKE_FOURCC('R', 'G', 'B', '4'), GST_MAKE_FOURCC('R', 'G', 'B', '4'), 32},
    {FOURCC_NONE, 0}
};


/*=============================================================================
                             GLOBAL VARIABLES
=============================================================================*/
/* None */

/*=============================================================================
                        LOCAL FUNCTION PROTOTYPES
=============================================================================*/

GST_DEBUG_CATEGORY_STATIC(mfw_gst_isink_debug);
static void mfw_gst_isink_base_init(gpointer);
static void mfw_gst_isink_class_init(MfwGstISinkClass *);
static void mfw_gst_isink_init(MfwGstISink *,
                                 MfwGstISinkClass *);

static void mfw_gst_isink_get_property(GObject *, 
                                         guint, GValue *,
                                         GParamSpec *);
static void mfw_gst_isink_set_property(GObject *, 
                                         guint, const GValue *,
                                         GParamSpec *);

static GstStateChangeReturn mfw_gst_isink_change_state
    (GstElement *, GstStateChange);

static gboolean mfw_gst_isink_setcaps(GstBaseSink *, GstCaps *);

static GstFlowReturn mfw_gst_isink_show_frame
                            (GstBaseSink *, GstBuffer *);

static GstFlowReturn mfw_gst_isink_buffer_alloc(GstBaseSink * bsink, 
                                                  guint64 offset,
                                                  guint size, GstCaps * caps,
                                                  GstBuffer ** buf);

/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/


static int fmt2bpp(ISinkFmt * fmttable, guint afmt)
{
    ISinkFmt * fmt = fmttable;
    while(fmt->fourcc!=FOURCC_NONE){
        if (fmt->fourcc==afmt){
            return fmt->bpp;
        }
        fmt++;
    };
    return 0;
}

static guint bpp2fmt(ISinkFmt * fmttable, gint abpp)
{
    ISinkFmt * fmt = fmttable;
    while(fmt->fourcc!=FOURCC_NONE){
        if (fmt->bpp==abpp){
            return fmt->fourcc;
        }
        fmt++;
    };
    return FOURCC_NONE;
}

static guint fmt2fmt(ISinkFmt * fmttable, gint afmt)
{
    ISinkFmt * fmt = fmttable;
    while(fmt->fourcc!=FOURCC_NONE){
        if (fmt->fourcc==afmt){
            return fmt->ifmt;
        }
        fmt++;
    };
    return 0;
}




#ifdef USE_X11
/*=============================================================================
FUNCTION:           mfw_gst_isink_set_xwindow_id

DESCRIPTION:        This function handle the set_xwindow_id event.

ARGUMENTS PASSED:
        overlay     -  Pointer to GstXOverlay
        xwindow_id  - Pointer to XID


RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_set_xwindow_id(GstXOverlay * overlay, XID xwindow_id)
{
    MfwGstISink *isink = MFW_GST_ISINK(overlay);
    GstXInfo *gstXInfo;
    /* If we already use that window return */
    if (xwindow_id == 0) {
        ISINK_ERROR("invalid window id.\n");
        return;
    }

    if (isink->gstXInfo == NULL) {
        isink->gstXInfo = mfw_gst_xinfo_new();
        isink->gstXInfo->parent = (void *)isink;
    }
    
    if (isink->gstXInfo->xcontext == NULL) {
        isink->gstXInfo->xcontext = mfw_gst_x11_xcontext_get();
        if (isink->gstXInfo->xcontext == NULL) {
            g_print("could not open display\n");
            mfw_gst_xinfo_free(isink->gstXInfo);
            return;
        }
        mfw_gst_xwindow_create(isink->gstXInfo, xwindow_id);
    }

    /* Enable the x11 capabilities */
    isink->x11enabled = TRUE;

    gstXInfo = isink->gstXInfo;
    isink->setXid = FALSE;
    /* If we already use that window return */
    if(gstXInfo->xwindow) {
        if (gstXInfo->xwindow->win == xwindow_id) {
            /* Handle all the events in the threads */
            isink->setXid = TRUE;
            return;
        }
    }

    return;
}


/*=============================================================================
FUNCTION:           mfw_gst_isink_expose

DESCRIPTION:        This function handle the expose event.

ARGUMENTS PASSED:
        overlay  -  Pointer to GstXOverlay


RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_expose (GstXOverlay * overlay)
{

    MfwGstISink *isink = MFW_GST_ISINK(overlay);
    VSConfig config;
    config.length = 0;
    config.data = NULL;
    configVideoSurface(isink->ocfg[0].vshandle, CONFIG_LAYER, &config);

#if 0
    PARAM_SET param = PARAM_NULL;

    if (!v4l_info->flow_lock)
        return;
    g_mutex_lock(v4l_info->flow_lock);
    param = mfw_gst_xv4l2_get_geometry(v4l_info);
    v4l_info->setpara |= param; // PARAM_SET_V4L | PARAM_SET_COLOR_KEY;
    g_mutex_unlock(v4l_info->flow_lock);
#endif
    GST_DEBUG("%s invoked\n",__FUNCTION__);
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_set_event_handling

DESCRIPTION:        This function set the X window events.

ARGUMENTS PASSED:
        overlay  -  Pointer to GstXOverlay
        handle_events - TRUE/FALSE


RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static void
mfw_gst_isink_set_event_handling (GstXOverlay * overlay,
    gboolean handle_events)
{
    MfwGstISink *isink = MFW_GST_ISINK(overlay);
    GstXInfo *gstXInfo = isink->gstXInfo;


    g_print("%s: handle events:%d.\n",__FUNCTION__, handle_events);
    gstXInfo->handle_events = handle_events;


    if (G_UNLIKELY (!gstXInfo->xwindow->win)) {
        return;
    }

    g_mutex_lock (gstXInfo->x_lock);

    if (handle_events) {
      XSelectInput (gstXInfo->xcontext->disp, gstXInfo->xwindow->win,
          ExposureMask | StructureNotifyMask | PointerMotionMask |
          KeyPressMask | KeyReleaseMask);
    } else {
        XSelectInput (gstXInfo->xcontext->disp, gstXInfo->xwindow->win, 0);
    }

    g_mutex_unlock (gstXInfo->x_lock);

    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_xoverlay_init

DESCRIPTION:        This function set the X window events.

ARGUMENTS PASSED:
        iface  -  Pointer to GstXOverlayClass


RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_xoverlay_init (GstXOverlayClass * iface)
{
  iface->set_xwindow_id = mfw_gst_isink_set_xwindow_id;
  iface->expose = mfw_gst_isink_expose;
  iface->handle_events = mfw_gst_isink_set_event_handling;

}

/*=============================================================================
FUNCTION:           mfw_gst_isink_got_xwindow_id

DESCRIPTION:        This function decorate the window.

ARGUMENTS PASSED:
        v4l_info  -  Pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_isink_got_xwindow_id (MfwGstISink *isink)
{
    gst_x_overlay_prepare_xwindow_id (GST_X_OVERLAY (isink));
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_interface_supported

DESCRIPTION:        This function decorate the window.

ARGUMENTS PASSED:
        iface  -  Pointer to GstImplementsInterface
        type   -  Pointer to GType

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean
mfw_gst_isink_interface_supported (GstImplementsInterface * iface, GType type)
{
  g_assert ( (type == GST_TYPE_X_OVERLAY) || (type == GST_TYPE_NAVIGATION));
  return TRUE;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_interface_init

DESCRIPTION:        This function decorate the window.

ARGUMENTS PASSED:
        klass  -  Pointer to GstImplementsInterfaceClass

RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_interface_init (GstImplementsInterfaceClass * klass)
{
  klass->supported = mfw_gst_isink_interface_supported;
}


static void
mfw_gst_isink_navigation_send_event (GstNavigation * navigation,
    GstStructure * structure)
{
    MfwGstISink *isink = MFW_GST_ISINK(navigation);
      GstPad *peer;

  GST_INFO("send the navigation event.\n");
  if ((peer = gst_pad_get_peer (GST_VIDEO_SINK_PAD (isink)))) {
    GstEvent *event;
    GstVideoRectangle src, dst, result;
    gdouble x, y, xscale = 1.0, yscale = 1.0;

    event = gst_event_new_navigation (structure);


    /* We take the flow_lock while we look at the window */

    if (!isink->gstXInfo->xwindow->win) {
      return;
    }

    /* We get the frame position using the calculated geometry from _setcaps
       that respect pixel aspect ratios */
    src.w = isink->icfg.srcfmt.croprect.width;
    src.h = isink->icfg.srcfmt.croprect.height; 
    dst.w = isink->ocfg[ISINK_CONFIG_LCD].desfmt.rect.right-isink->ocfg[ISINK_CONFIG_LCD].desfmt.rect.left;
    dst.h = isink->ocfg[ISINK_CONFIG_LCD].desfmt.rect.bottom-isink->ocfg[ISINK_CONFIG_LCD].desfmt.rect.top;


    if (0){//!isink->stretch) {
      gst_video_sink_center_rect (src, dst, &result, TRUE);
    } else {
      result.x = result.y = 0;
      result.w = dst.w;
      result.h = dst.h;
    }

    /* We calculate scaling using the original video frames geometry to include
       pixel aspect ratio scaling. */
    xscale = (gdouble) isink->icfg.srcfmt.croprect.width / result.w;
    yscale = (gdouble) isink->icfg.srcfmt.croprect.height/ result.h;

    /* Converting pointer coordinates to the non scaled geometry */
    if (gst_structure_get_double (structure, "pointer_x", &x)) {
      x = MIN (x, result.x + result.w);
      x = MAX (x - result.x, 0);
      gst_structure_set (structure, "pointer_x", G_TYPE_DOUBLE,
          (gdouble) x * xscale, NULL);
    }
    if (gst_structure_get_double (structure, "pointer_y", &y)) {
      y = MIN (y, result.y + result.h);
      y = MAX (y - result.y, 0);
      gst_structure_set (structure, "pointer_y", G_TYPE_DOUBLE,
          (gdouble) y * yscale, NULL);
    }

    gst_pad_send_event (peer, event);
    gst_object_unref (peer);
  }
}

static void
mfw_gst_isink_navigation_init (GstNavigationInterface * iface)
{
  iface->send_event = mfw_gst_isink_navigation_send_event;
}

#endif

static void
mfw_gst_isink_buffer_finalize(MFWGstIBuf *ibuf)
{
    if (ibuf->priv){
        freeHwBuffer(ibuf->priv);
        ibuf->priv=NULL;
    }
}


/*=============================================================================
FUNCTION:           mfw_gst_isink_buffer_init    
        
DESCRIPTION:        This funtion initialises the buffer class of the V4lsink
                    plug-in

ARGUMENTS PASSED:
        ibuf -   pointer to Isink buffer class
        g_class        -   global pointer

  
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_buffer_init(MFWGstIBuf * ibuf,
			    gpointer g_class)
{
    ibuf->priv = NULL;
    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_buffer_class_init    
        
DESCRIPTION:        This funtion registers the  funtions used by the 
                    buffer class of the V4lsink plug-in

ARGUMENTS PASSED:
        g_class        -   class from which the mini objext is derived
        class_data     -   global class data
  
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_buffer_class_init(gpointer g_class, gpointer class_data)
{
    GstMiniObjectClass *mini_object_class = GST_MINI_OBJECT_CLASS(g_class);
    mini_object_class->finalize = (GstMiniObjectFinalizeFunction)mfw_gst_isink_buffer_finalize;
    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_buffer_get_type    
        
DESCRIPTION:        This funtion registers the  buffer class 
                    on to the Isink plugin

ARGUMENTS PASSED:   None
  
RETURN VALUE:       return the registered buffer class
      
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

GType mfw_gst_isink_buffer_get_type(void)
{

    static GType _mfw_gst_isink_buffer_type;

    if (G_UNLIKELY(_mfw_gst_isink_buffer_type == 0)) {
	static const GTypeInfo isink_buffer_info = {
	    sizeof(GstBufferClass),
	    NULL,
	    NULL,
	    mfw_gst_isink_buffer_class_init,
	    NULL,
	    NULL,
	    sizeof(MFWGstIBuf),
	    0,
	    (GInstanceInitFunc) mfw_gst_isink_buffer_init,
	    NULL
	};
	_mfw_gst_isink_buffer_type =
	    g_type_register_static(GST_TYPE_BUFFER, "MFWGstIBuf",
				   &isink_buffer_info, 0);
    }
    
    return _mfw_gst_isink_buffer_type;
}

void mfw_gst_isink_close(MfwGstISink* isink)
{
    MFWGstIBuf * ibuf;
    int i;

    for (i=0;i<ISINK_CONFIG_MAX;i++){
        if (isink->ocfg[i].vshandle){
            destroyVideoSurface(isink->ocfg[i].vshandle);
            isink->ocfg[i].vshandle = NULL;
        }
    }

    if (isink->hwbufhandle)
        setHWBufferAllocatorSelfDescontruct(isink->hwbufhandle);
        isink->hwbufhandle=NULL;/* IT WILL DESTROY ITSELF */
    
    if (isink->curbuf){
        gst_buffer_unref(isink->curbuf);
        isink->curbuf = NULL;
    }

    isink->closed = TRUE;
}

static MFWGstIBuf *createIsinkBuf(MfwGstISink *isink, int size)
{
    MFWGstIBuf * ibuf=NULL;
    char * data;
    ibuf = (MFWGstIBuf *)gst_mini_object_new(MFW_GST_TYPE_ISINK_BUFFER);
    if (isink->hwbufhandle==NULL){
        isink->hwbufhandle=createHWBufferAllocator();
    }
    ibuf->priv = newHwBuffer(isink->hwbufhandle,size);
    if (ibuf->priv==NULL){
        (GST_BUFFER_DATA(GST_BUFFER_CAST(ibuf))) = g_malloc(size);
        (GST_BUFFER_MALLOCDATA(GST_BUFFER_CAST(ibuf))) = (GST_BUFFER_DATA(GST_BUFFER_CAST(ibuf)));
        if ((GST_BUFFER_DATA(GST_BUFFER_CAST(ibuf)))==NULL){
            goto error;
        }
        GST_BUFFER_FLAG_UNSET(&ibuf->gstBuf, GST_BUFFER_FLAG_LAST);
    }else{
        GST_BUFFER_DATA(ibuf) = ((HWBufDesc *)ibuf->priv)->virtaddr;
        GST_BUFFER_OFFSET(ibuf) = ((HWBufDesc *)ibuf->priv)->phyaddr;
        GST_BUFFER_FLAG_SET(&ibuf->gstBuf, GST_BUFFER_FLAG_LAST);
    }
    GST_BUFFER_SIZE(ibuf)=size;
    return ibuf;
error:
    if (ibuf)
        gst_buffer_unref(ibuf);
    return NULL;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_buffer_alloc   
        
DESCRIPTION:        This function initailise the sl driver
                    and gets the new buffer for display             

ARGUMENTS PASSED:  
          bsink :   pointer to GstBaseSink
		  buf   :   pointer to new GstBuffer
		  size  :   size of the new buffer
          offset:   buffer offset
		  caps  :   pad capability
        
RETURN VALUE:       GST_FLOW_OK/GST_FLOW_ERROR
      
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstFlowReturn
mfw_gst_isink_buffer_alloc(GstBaseSink * bsink, guint64 offset,
				     guint size, GstCaps * caps,
				     GstBuffer ** buf)
{
    GstBuffer *newbuf = NULL;
    MfwGstISink *isink = MFW_GST_ISINK(bsink);
    MFWGstIBuf *ibuf = NULL;
    GstStructure *s = NULL;
    
    if (G_UNLIKELY(isink->init == FALSE)) 
    {
        gint right=0, bottom=0;
        guint fmt;
        int i;
        
        s = gst_caps_get_structure(caps, 0);

        InCfg * icfg = &isink->icfg;

        memset(icfg, 0, sizeof(InCfg));

        gst_structure_get_int(s, "width", &icfg->srcfmt.croprect.width);
        gst_structure_get_int(s, "height", &icfg->srcfmt.croprect.height);
        gst_structure_get_fraction(s, "pixel-aspect-ratio", &icfg->aspectRatioN,
				   &icfg->aspectRatioD);
        
        gst_structure_get_int(s, "crop-left-by-pixel", &icfg->srcfmt.croprect.win.left);
        gst_structure_get_int(s, "crop-top-by-pixel", &icfg->srcfmt.croprect.win.top);
        gst_structure_get_int(s, "crop-right-by-pixel", &right);
        gst_structure_get_int(s, "crop-bottom-by-pixel", &bottom);

        icfg->srcfmt.croprect.win.left += isink->cleft;
        icfg->srcfmt.croprect.win.top += isink->ctop;
        right += isink->cright;
        bottom += isink->cbottom;
        icfg->srcfmt.croprect.win.right = icfg->srcfmt.croprect.width-right;
        icfg->srcfmt.croprect.win.bottom = icfg->srcfmt.croprect.height-bottom;
       
        gst_structure_get_int(s, "num-buffers-required", &icfg->requestBufferNum);

        if (icfg->requestBufferNum<2)
            icfg->requestBufferNum=2;
        
        fmt=FOURCC_NONE;
        gst_structure_get_fourcc(s, "format", &fmt);

        gint bpp;
        if (fmt==FOURCC_NONE){
            gst_structure_get_int(s, "bpp", &bpp);
            fmt = bpp2fmt(isink_rgb_fmts, bpp);
        }else{
            bpp = fmt2bpp(isink_yuv_fmts,fmt);
            if (bpp==0){
                bpp = fmt2bpp(isink_rgb_fmts,fmt);
            }
        }

        icfg->srcfmt.fmt = fmt2fmt(isink_yuv_fmts, fmt);
        if (icfg->srcfmt.fmt==0)
            icfg->srcfmt.fmt = fmt2fmt(isink_rgb_fmts, fmt);

        icfg->framebufsize = icfg->srcfmt.croprect.width * icfg->srcfmt.croprect.height * bpp / 8;
 
        for (i=0;i<ISINK_CONFIG_MAX;i++){
            OutCfg * ocfg = &isink->ocfg[i];
            if (ocfg->enabled){
                if (RECT_WIDTH(&ocfg->desfmt.rect)==0)
                    ocfg->desfmt.rect.right=320;
                if (RECT_HEIGHT(&ocfg->desfmt.rect)==0)
                    ocfg->desfmt.rect.bottom=240;
                ocfg->vshandle = createVideoSurface(ocfg->fbid, &isink->icfg.srcfmt, &ocfg->desfmt);
            }
        }

        isink->init = TRUE;
    }

    ibuf = createIsinkBuf(isink, isink->icfg.framebufsize);
    if (ibuf == NULL) {
        g_print("\n!!>>I_SINK: Could not allocate buffer\n");
        *buf = NULL;
        
        return GST_FLOW_ERROR;
    } else {
        GST_BUFFER_SIZE(ibuf) = size;
    	newbuf = GST_BUFFER_CAST(ibuf);
    	gst_buffer_set_caps(newbuf, caps);
    	*buf = newbuf;
    	return GST_FLOW_OK;
    }
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_set_property   
        
DESCRIPTION:        This function is notified if application changes the 
                    values of a property.            

ARGUMENTS PASSED:
        object  -   pointer to GObject   
        prop_id -   id of element
        value   -   pointer to Gvalue
        pspec   -   pointer to GParamSpec
        
RETURN VALUE:       GST_FLOW_OK/GST_FLOW_ERROR
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_set_property(GObject * object, guint prop_id,
			     const GValue * value, GParamSpec * pspec)
{
    MfwGstISink *isink = MFW_GST_ISINK(object);
    OutCfg * lcdcfg = &isink->ocfg[ISINK_CONFIG_LCD];
    gint intvalue;
    switch (prop_id){
        case ISINK_PROP_ROTATION_0:
            lcdcfg->desfmt.rot = g_value_get_int(value);
            break;
        case ISINK_PROP_DISPWIN_LEFT_0:
            intvalue = g_value_get_int(value);
            lcdcfg->desfmt.rect.right += (intvalue-lcdcfg->desfmt.rect.left);
            lcdcfg->desfmt.rect.left = intvalue;
            break;
        case ISINK_PROP_DISPWIN_TOP_0:
            intvalue = g_value_get_int(value);
            lcdcfg->desfmt.rect.bottom += (intvalue-lcdcfg->desfmt.rect.top);
            lcdcfg->desfmt.rect.top = intvalue;
            break;
        case ISINK_PROP_DISPWIN_WIDTH_0:
            intvalue = g_value_get_int(value);
            lcdcfg->desfmt.rect.right = lcdcfg->desfmt.rect.left+intvalue;
            break;
        case ISINK_PROP_DISPWIN_HEIGHT_0:
            intvalue = g_value_get_int(value);
            lcdcfg->desfmt.rect.bottom = lcdcfg->desfmt.rect.top+intvalue;
            break;
        
        case ISINK_PROP_INPUT_CROP_LEFT_0:
            isink->cleft = g_value_get_int(value);
            break;
        case ISINK_PROP_INPUT_CROP_RIGHT_0:
            isink->cright= g_value_get_int(value);
            break;        
        case ISINK_PROP_INPUT_CROP_TOP_0:
            isink->ctop= g_value_get_int(value);
            break;        
        case ISINK_PROP_INPUT_CROP_BOTTOM_0:
            isink->cbottom= g_value_get_int(value);
            break;        
        default:
            break;
    }
    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_get_property    
        
DESCRIPTION:        This function is notified if application requests the 
                    values of a property.                  

ARGUMENTS PASSED:
        object  -   pointer to GObject   
        prop_id -   id of element
        value   -   pointer to Gvalue
        pspec   -   pointer to GParamSpec
  
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_get_property(GObject * object, guint prop_id,
			     GValue * value, GParamSpec * pspec)
{

    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_show_frame   
        
DESCRIPTION:        Process data to display      

ARGUMENTS PASSED:
        pad -   pointer to GstPad;
        buf -   pointer to GstBuffer
        
RETURN VALUE:       GST_FLOW_OK/GST_FLOW_ERROR
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gint diff_time(struct timeval * past, struct timeval * now)
{
    return (gint)(((gint64)(now->tv_sec)-(gint64)(past->tv_sec))*(gint64)(1000000)+
        ((gint64)(now->tv_usec)-(gint64)(past->tv_usec)));
}
    
static GstFlowReturn mfw_gst_isink_show_frame(GstBaseSink * basesink,
                                                GstBuffer * gstbuf)
{
    MfwGstISink *isink = MFW_GST_ISINK(basesink);
    OutCfg * ocfg = &isink->ocfg[0];
    SourceFrame frame;
    GstFlowReturn ret = GST_FLOW_OK;

#if 0    
    struct timeval current;
    //g_print(YELLOW_STR("%s %d %p %p\n", __FUNCTION__, __LINE__, buf, GST_BUFFER_OFFSET(buf)));
    gettimeofday(&current, NULL);
    isink->stat.displayed++;
    
    gst_buffer_ref(buf);
    
    if (isink->curbuf){
        gst_buffer_unref(isink->curbuf);
        
    }
    isink->curbuf = buf;
    if (isink->stat.started){
        gint interval = diff_time(&isink->stat.lastshow, &current);
        if (isink->stat.mininterval==0){
            isink->stat.mininterval=isink->stat.maxinterval=interval;
        }else{
            if (interval<isink->stat.mininterval)
                isink->stat.mininterval=interval;
            if (interval>isink->stat.maxinterval)
                isink->stat.maxinterval=interval;
        }
        isink->stat.total+=interval;
    }else{
        
        isink->stat.started = TRUE;
    }
    isink->stat.lastshow = current;
    
#else
    if (isink->curbuf){
            gst_buffer_unref(isink->curbuf);
            isink->curbuf=NULL;
                
    }

    if (GST_BUFFER_FLAG_IS_SET(gstbuf,GST_BUFFER_FLAG_LAST)){
        gst_buffer_ref(gstbuf);


        isink->curbuf = gstbuf;

        if (MFW_GST_IS_ISINK_BUFFER(gstbuf)){
            MFWGstIBuf * ibuf = (MFWGstIBuf *)gstbuf;
            frame.paddr = ((HWBufDesc *)ibuf->priv)->phyaddr;
        }else{
            frame.paddr = GST_BUFFER_OFFSET(gstbuf);
        }
    }else{
        MFWGstIBuf * ibuf; 
        if ((ret = mfw_gst_isink_buffer_alloc(isink, 0, GST_BUFFER_SIZE(gstbuf), GST_BUFFER_CAPS(gstbuf), &ibuf))!=GST_FLOW_OK){
            GST_ERROR("Can not allocate ibuf for non-dma buffer, ret=%d", ret);
            goto done;
        }
        if (GST_BUFFER_FLAG_IS_SET(ibuf,GST_BUFFER_FLAG_LAST)){
            ISINK_MEMCPY(GST_BUFFER_DATA(ibuf), GST_BUFFER_DATA(gstbuf), GST_BUFFER_SIZE(gstbuf));
            frame.paddr = ((HWBufDesc *)ibuf->priv)->phyaddr;
            isink->curbuf = ibuf;
        }else{/* nothing to render */
            gst_buffer_unref(ibuf);
            goto done;
        }
        
    }

    if (ocfg->vshandle==NULL){
        MFWGstIBuf * ibuf; 
        if ((ret = mfw_gst_isink_buffer_alloc(isink, 0, GST_BUFFER_SIZE(gstbuf), GST_BUFFER_CAPS(gstbuf), &ibuf))!=GST_FLOW_OK){
            GST_ERROR("Can not allocate ibuf for non-dma buffer, ret=%d", ret);
            goto done;
        }
        gst_buffer_unref(ibuf);
    }
    render2VideoSurface(ocfg->vshandle, &frame, NULL);
#endif  
done:
    return GST_FLOW_OK;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_setcaps
         
DESCRIPTION:        This function does the capability negotiation between adjacent pad  

ARGUMENTS PASSED:    
        basesink    -   pointer to isink
        vscapslist  -   pointer to GstCaps
          
RETURN VALUE:       TRUE or FALSE depending on capability is negotiated or not.
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
   	    
=============================================================================*/

static gboolean mfw_gst_isink_setcaps(GstBaseSink * basesink,
                                        GstCaps * vscapslist)
{
    MfwGstISink * isink = MFW_GST_ISINK(basesink);
    guint32 format = 0;
    GstStructure *structure = NULL;

    structure = gst_caps_get_structure(vscapslist, 0);
    {
        gint sfd_val = 0;
        gboolean ret;

        ret = gst_structure_get_int(structure,"sfd",&sfd_val);
        if (ret == TRUE) {
            GST_DEBUG("sfd = %d.\n",sfd_val);
            if (sfd_val == 1)
                basesink->abidata.ABI.max_lateness = -1;
        }else {
            GST_DEBUG(">>I_SINK: no sfd field found in caps.\n");
        }
    }

    
#ifdef USE_X11
    /* Send the got-xwindow-id message to request 
     * the set-xwindow-id callback.
     */
    mfw_gst_isink_got_xwindow_id(isink);
    if (1){
        gint timeout=20; //timeout 2s
        while ((isink->gstXInfo==NULL)&& (timeout-->0)){//((isink->disp_height<16) && (timeout-->0)) {
            usleep(200000);
            
            //mfw_gst_xisink_refresh_geometry(isink);

        }

        
        if ((isink->gstXInfo) && (isink->gstXInfo->running == FALSE)) {
            isink->gstXInfo->running = TRUE;
            isink->gstXInfo->event_thread = g_thread_create (
                    (GThreadFunc) mfw_gst_xisink_event_thread, isink, TRUE, NULL);
        }else{
            ISINK_ERROR("can not create thread%s:%d\n", __FUNCTION__, __LINE__);
        }

    }
    else {
    }
    
#endif    

    return TRUE;
}

/*=============================================================================
FUNCTION:           mfw_gst_isink_change_state   
        
DESCRIPTION:        This function keeps track of different states of pipeline.        

ARGUMENTS PASSED:
        element     -   pointer to element 
        transition  -   state of the pipeline       
  
RETURN VALUE:
        GST_STATE_CHANGE_FAILURE    - the state change failed  
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded  
        GST_STATE_CHANGE_ASYNC      - the state change will happen asynchronously  
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled  
      
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstStateChangeReturn mfw_gst_isink_change_state(GstElement * element, 
                                                         GstStateChange transition) 
{
    MfwGstISink *isink = MFW_GST_ISINK(element);

    GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
    guint8 index;
    switch (transition) {
        case GST_STATE_CHANGE_NULL_TO_READY:
            isink->init = FALSE;
            if (isink->hwbufhandle==NULL){
                isink->hwbufhandle=createHWBufferAllocator();
            }

        break;
        case GST_STATE_CHANGE_READY_TO_PAUSED:
        break;
        case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
        break;
        default:
        break;
    }

    ret = GST_ELEMENT_CLASS(isink->parent_class)->
    change_state(element, transition);

    switch (transition) {
        case GST_STATE_CHANGE_PAUSED_TO_READY:
            g_print("trans change to ready\n");
	    break;
        case GST_STATE_CHANGE_READY_TO_NULL:
        {
#ifdef USE_X11

            // mfw_gst_xv4l2_clear_color(v4l_info);

            /* Set the running flag to false and wait for the thread exit */
            if (isink->gstXInfo){
            isink->gstXInfo->running = FALSE;
            if (isink->x11enabled) {
                g_thread_join(isink->gstXInfo->event_thread);
            }
            if (isink->gstXInfo->xwindow) {
                mfw_gst_xwindow_destroy(isink->gstXInfo, isink->gstXInfo->xwindow);
                isink->gstXInfo->xwindow = NULL;
            }            
            mfw_gst_xcontext_free(isink->gstXInfo);

            mfw_gst_xinfo_free(isink->gstXInfo);
            isink->gstXInfo = NULL;
            }
            isink->x11enabled = FALSE;
            

#endif
            g_print("trans change to null\n");
            mfw_gst_isink_close(isink);
            

            isink->init = FALSE;

            
            
            break;
        }
        default:			/* do nothing */
        break;
    }
    return ret;
}
void mfw_gst_fb0_set_colorkey(MfwGstISink * isink)
{
    gboolean ret = TRUE;
#ifndef _MX233
    struct mxcfb_color_key colorKey;
    struct fb_var_screeninfo fbVar;

    int fb = open("/dev/fb0", O_RDWR, 0);
    if (fb<=0)
        return;

    if (ioctl(fb, FBIOGET_VSCREENINFO, &fbVar) < 0) {
        g_print("get vscreen info failed.\n");
        ret = FALSE;
    }

    if (fbVar.bits_per_pixel == 16) {
        isink->colorkey= RGB888TORGB565(RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE));
        GST_DEBUG("%08X:%08X:%8X\n",RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE),
            RGB888TORGB565(RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE)), 
            RGB565TOCOLORKEY(RGB888TORGB565(RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE))));
        colorKey.color_key = RGB565TOCOLORKEY(isink->colorkey);
    }
    else if ((fbVar.bits_per_pixel == 32) || (fbVar.bits_per_pixel == 24)) {
        isink->colorkey = RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE);
        colorKey.color_key = isink->colorkey;

    }

    colorKey.enable = 1;
    if (ioctl(fb, MXCFB_SET_CLR_KEY, &colorKey) < 0) {
        g_print("set color key failed.\n");
        ret = FALSE;
    }
#if 1    
    struct mxcfb_gbl_alpha g_alpha;
    g_alpha.alpha = 255;
    g_alpha.enable = 1;
    
    if (ioctl(fb, MXCFB_SET_GBL_ALPHA, &g_alpha) < 0) {
        g_print("set color key failed.\n");
        ret = FALSE;
    }
#endif
    g_print(RED_STR("set color key\n"));
    close(fb);
#endif
}


/*=============================================================================
FUNCTION:           mfw_gst_isink_init   
        
DESCRIPTION:        Create the pad template that has been registered with the 
                    element class in the _base_init and do library table 
                    initialization      

ARGUMENTS PASSED:
        isink  -    pointer to isink element structure      
  
RETURN VALUE:       NONE
PRE-CONDITIONS:     _base_init and _class_init are called 
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_isink_init(MfwGstISink * isink,
				                 MfwGstISinkClass * klass)
{
    OutCfg * ocfg;
    
    isink->parent_class = g_type_class_peek_parent(klass);
    
    memset(&isink->icfg, 0, sizeof(InCfg));
    memset(isink->ocfg, 0, sizeof(OutCfg)*ISINK_CONFIG_MAX);
    ocfg = &isink->ocfg[ISINK_CONFIG_LCD];

    ocfg->enabled = TRUE;
    ocfg->desfmt.rect.left = ocfg->desfmt.rect.top = 0;
    ocfg->desfmt.rect.right = 320;
    ocfg->desfmt.rect.bottom = 240;
    ocfg->fbid = LCD_FB_NUM;

    isink->curbuf = NULL;

    isink->hwbufhandle=createHWBufferAllocator();
    mfw_gst_fb0_set_colorkey(isink);
#define MFW_GST_ISINK_PLUGIN VERSION
    PRINT_PLUGIN_VERSION(MFW_GST_ISINK_PLUGIN);
    return;
}

static void mfw_gst_isink_finalize(GObject *object)
{
    MfwGstISink * isink = MFW_GST_ISINK(object);
    
    if (isink->hwbufhandle){
        destroyHWBufferAllocator(isink->hwbufhandle);
        isink->hwbufhandle = NULL;
    }
#if 0
    g_print("static total frame %d, average time %fms, maxtime %dms, mintime %dms\n",
        isink->stat.displayed, (float)(isink->stat.total)/(float)1000/(float)(isink->stat.displayed-1), 
        isink->stat.maxinterval/1000, isink->stat.mininterval/1000);
     #endif   
}


/*=============================================================================
FUNCTION:           mfw_gst_isink_class_init    
        
DESCRIPTION:        Initialise the class only once (specifying what signals,
                    arguments and virtual functions the class has and 
                    setting up global state)    

ARGUMENTS PASSED:
            klass   -   pointer to mp3decoder element class
  
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_isink_class_init(MfwGstISinkClass * klass)
{

    GObjectClass *gobject_class;
    GstElementClass *gstelement_class;
    GstBaseSinkClass *gstvs_class;

    gobject_class = (GObjectClass *) klass;
    gstelement_class = (GstElementClass *) klass;
    gstvs_class = (GstBaseSinkClass *) klass;

    gstelement_class->change_state =
	GST_DEBUG_FUNCPTR(mfw_gst_isink_change_state);

    gobject_class->set_property = mfw_gst_isink_set_property;

    gobject_class->get_property = mfw_gst_isink_get_property;
    gobject_class->finalize = mfw_gst_isink_finalize;

    gstvs_class->set_caps = GST_DEBUG_FUNCPTR(mfw_gst_isink_setcaps);
    gstvs_class->render = GST_DEBUG_FUNCPTR(mfw_gst_isink_show_frame);
    gstvs_class->buffer_alloc = GST_DEBUG_FUNCPTR(mfw_gst_isink_buffer_alloc);

    
    g_object_class_install_property(gobject_class, ISINK_PROP_DISPWIN_WIDTH_0,
                        g_param_spec_int("disp-width",
                                 "Disp_Width",
                                 "get/set the width of the image to be displayed",
                                 0, G_MAXINT, 0,
                                 G_PARAM_READWRITE));
    
    g_object_class_install_property(gobject_class, ISINK_PROP_DISPWIN_HEIGHT_0,
                    g_param_spec_int("disp-height",
                             "Disp_Height",
                             "get/set the height of the image to be displayed",
                             0, G_MAXINT, 0,
                             G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, ISINK_PROP_DISPWIN_LEFT_0,
                    g_param_spec_int("axis-left",
                             "axis-left",
                             "gets the left co-ordinate of the origin of display",
                             0, G_MAXINT, 0,
                             G_PARAM_READWRITE));
    
    g_object_class_install_property(gobject_class, ISINK_PROP_DISPWIN_TOP_0,
                    g_param_spec_int("axis-top",
                             "axis-top",
                             "gets the top co-ordinate of the origin of display",
                             0, G_MAXINT, 0,
                             G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, ISINK_PROP_INPUT_CROP_LEFT_0,
                    g_param_spec_int("crop-left",
                             "crop-left",
                             "get/set the height of the image to be displayed",
                             0, G_MAXINT, 0,
                             G_PARAM_READWRITE));
    g_object_class_install_property(gobject_class, ISINK_PROP_INPUT_CROP_RIGHT_0,
                    g_param_spec_int("crop-right",
                             "crop-right",
                             "get/set the height of the image to be displayed",
                             0, G_MAXINT, 0,
                             G_PARAM_READWRITE));
    g_object_class_install_property(gobject_class, ISINK_PROP_INPUT_CROP_TOP_0,
                    g_param_spec_int("crop-top",
                             "crop-top",
                             "get/set the height of the image to be displayed",
                             0, G_MAXINT, 0,
                             G_PARAM_READWRITE));
    g_object_class_install_property(gobject_class, ISINK_PROP_INPUT_CROP_BOTTOM_0,
                g_param_spec_int("crop-bottom",
                         "crop-bottom",
                         "get/set the height of the image to be displayed",
                         0, G_MAXINT, 0,
                         G_PARAM_READWRITE));

    /* FixME: The i.MX233 does not support rotate */
#ifndef _MX233 
    g_object_class_install_property(gobject_class, ISINK_PROP_ROTATION_0,
                    g_param_spec_int("rotate", "Rotate",
                             "gets the angle at which display is to be rotated",
                             0, G_MAXINT, 0,
                             G_PARAM_READWRITE));
#endif

    g_object_class_install_property(gobject_class, ISINK_PROP_SETPARA,
                   g_param_spec_int ("setpara", "Setpara",
                             "set parameter of V4L2, 1: Set V4L 2: Set Color",
                             0,3,0,
                             G_PARAM_READWRITE));
    
    return;

}


/*=============================================================================
FUNCTION:           mfw_gst_isink_base_init   
        
DESCRIPTION:       sl Sink element details are registered with the plugin during
                   _base_init ,This function will initialise the class and child 
                    class properties during each new child class creation       

ARGUMENTS PASSED:
            Klass   -   void pointer
  
RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/


static void mfw_gst_isink_base_init(gpointer g_class)
{
    GstElementClass *element_class = GST_ELEMENT_CLASS(g_class);
    GstCaps *capslist;
    GstPadTemplate *sink_template = NULL;
    ISinkFmt * fmt;
    
    /* make a list of all available caps */
    capslist = gst_caps_new_empty();

    /* add yuv formats */
    fmt = isink_yuv_fmts;
    while (fmt->fourcc!=FOURCC_NONE) {
    	gst_caps_append_structure(capslist,
            gst_structure_new("video/x-raw-yuv",
        	    "format", GST_TYPE_FOURCC, fmt->fourcc, 
                "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT, 
                "height", GST_TYPE_INT_RANGE, 1, G_MAXINT, 
                "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, 100, 1, 
                 NULL));
        fmt++;
    }

    
    /* add rgb formats */
    gst_caps_append_structure(capslist,
                  gst_structure_new("video/x-raw-rgb",
                  "bpp",GST_TYPE_INT_RANGE, 1,32,
                  "depth",GST_TYPE_INT_RANGE, 1,32,
                            NULL));

    sink_template = gst_pad_template_new("sink",
					 GST_PAD_SINK, GST_PAD_ALWAYS,
					 capslist);

    gst_element_class_add_pad_template(element_class, sink_template);
    gst_element_class_set_details(element_class, &mfw_gst_isink_details);

    GST_DEBUG_CATEGORY_INIT(mfw_gst_isink_debug, "mfw_isink", 0,
			    "screen layer video sink element");
    return;

}

/*=============================================================================
FUNCTION:           mfw_gst_isink_get_type    
        
DESCRIPTION:        Interfaces are initiated in this function.you can register one 
                    or more interfaces  after having registered the type itself.

ARGUMENTS PASSED:   None
  
RETURN VALUE:       A numerical value ,which represents the unique identifier 
                    of this element(isink)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
GType mfw_gst_isink_get_type(void)
{
    static GType mfwIsink_type = 0;

    if (!mfwIsink_type) {
	static const GTypeInfo mfwIsink_info = {
	    sizeof(MfwGstISinkClass),
	    mfw_gst_isink_base_init,
	    NULL,
	    (GClassInitFunc) mfw_gst_isink_class_init,
	    NULL,
	    NULL,
	    sizeof(MfwGstISink),
	    0,
	    (GInstanceInitFunc) mfw_gst_isink_init,
	};

	mfwIsink_type = g_type_register_static(GST_TYPE_VIDEO_SINK,
						 "MfwGstISink",
						 &mfwIsink_info, 0);

#ifdef USE_X11
    {
        static const GInterfaceInfo iface_info = {
        (GInterfaceInitFunc) mfw_gst_isink_interface_init,
        NULL,
        NULL,
        };

        static const GInterfaceInfo overlay_info = {
        (GInterfaceInitFunc) mfw_gst_isink_xoverlay_init,
        NULL,
        NULL,
        };

        static const GInterfaceInfo navigation_info = {
          (GInterfaceInitFunc) mfw_gst_isink_navigation_init,
          NULL,
          NULL,
        };

        g_type_add_interface_static (mfwIsink_type, GST_TYPE_IMPLEMENTS_INTERFACE,
            &iface_info);
        g_type_add_interface_static (mfwIsink_type, GST_TYPE_X_OVERLAY,
            &overlay_info);
        g_type_add_interface_static (mfwIsink_type, GST_TYPE_NAVIGATION,
            &navigation_info);
    }
#endif

    
    }

    GST_DEBUG_CATEGORY_INIT(mfw_gst_isink_debug, "mfw_isink",
			    0, "Isink");

    mfw_gst_isink_buffer_get_type();

    return mfwIsink_type;
}



/*=============================================================================
FUNCTION:           plugin_init

DESCRIPTION:        Special function , which is called as soon as the plugin or 
                    element is loaded and information returned by this function 
                    will be cached in central registry

ARGUMENTS PASSED:
        plugin     -    pointer to container that contains features loaded 
                        from shared object module

RETURN VALUE:
        return TRUE or FALSE depending on whether it loaded initialized any 
        dependency correctly

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean plugin_init(GstPlugin * plugin)
{
    if (!gst_element_register(plugin, "mfw_isink", FSL_GST_RANK_HIGH,
			      MFW_GST_TYPE_ISINK))
        return FALSE;

    return TRUE;
}

/*****************************************************************************/
/*    This is used to define the entry point and meta data of plugin         */
/*****************************************************************************/

GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,	/* major version of Gstreamer */
		  GST_VERSION_MINOR,	/* minor version of Gstreamer    */
		  "mfw_isink",	/* name of the plugin            */
		  "Video display plugin based on IPU",	/* what plugin actually does     */
		  plugin_init,	/* first function to be called   */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  FSL_GST_MM_PLUGIN_PACKAGE_NAME, FSL_GST_MM_PLUGIN_PACKAGE_ORIG)
