/*
 * Copyright (C) 2010 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    vss_common.c
 *
 * Description:    Implementation for ipu lib based render service.
 *
 * Portability:    This code is written for Linux OS.
 */  
 
/*
 * Changelog: 
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdint.h>

#include <semaphore.h>
#include <fcntl.h>

#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/time.h>

#include "mxc_ipu_hl_lib.h"
#include "video_surface.h"

#include "fsl_debug.h"

#include "vss_common.h"
#undef VSS_DAEMON


typedef struct{
    const char * devname;
    int fb_fd;
}FbDesc;


static FbDesc gFBDescs[] = {
    {"/dev/fb0", 0},
    {"/dev/fb1", 0},
    {"/dev/fb2", 0}
};

extern  VSLock * 	gVSlock;
extern  VideoSurfacesControl * gVSctl;


int _getDevicefd(VideoDevice * vd)
{
    int fd;
    if ((fd = gFBDescs[vd->fbidx].fb_fd)==0){
        fd = open(gFBDescs[vd->fbidx].devname, O_RDWR, 0);
        if (fd<=0)
            fd = 0;
        else
            gFBDescs[vd->fbidx].fb_fd = fd;
    }
    return fd;
}



VSLock * 
_getAndLockVSLock(int flag)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    VSLock * lock;

    int oflag = 0;
    if (flag & VS_IPC_CREATE)
        oflag |= O_CREAT;

    if (flag & VS_IPC_EXCL)
        oflag |= O_EXCL;

    printf("get lock %s\n", VS_LOCK_NAME);
    
    lock = sem_open(VS_LOCK_NAME, oflag, 0666  , 1);

    
    
	if(SEM_FAILED == lock){
        VS_ERROR("%s: can not get lock %s!\n", __FUNCTION__, VS_LOCK_NAME);
		goto err;
	}

    VS_LOCK(lock);
    
    return lock;
err:
    return NULL;
}



VideoSurfacesControl * _getVSControl(int flag)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    VideoSurfacesControl * control;
	int	shmid;
	struct	stat shmStat;


    int oflag = O_RDWR;
    if (flag & VS_IPC_CREATE)
        oflag |= O_CREAT;
    if (flag & VS_IPC_EXCL)
        oflag |= O_EXCL;

    shmid = shm_open(VS_SHMEM_NAME, oflag, 0666);

    if(shmid == -1){
        VS_ERROR("%s: can not get share memory %s!\n", __FUNCTION__, VS_SHMEM_NAME);
		goto err;
	}
    
	ftruncate(shmid,(off_t)(3 * sizeof(VideoSurfacesControl)));
	/* Connect to the shm */
	fstat(shmid, &shmStat);

	control = (VideoSurfacesControl *)mmap(NULL,shmStat.st_size,PROT_READ|PROT_WRITE,MAP_SHARED,shmid,0);
    if ((control==NULL) || (control==MAP_FAILED)){
        VS_ERROR("%s: can not mmap share memory %d!\n", __FUNCTION__, shmid);
        goto err;
    }


    return control;
err:
    return NULL;
}



int _initVideoDevice(VideoDevice * vd)
{
    struct fb_var_screeninfo fb_var;
    int fd = open(MAIN_DEVICE_NAME, O_RDWR, 0);
    
    if (fd>0){
        
        VS_IOCTL(fd, FBIOGET_VSCREENINFO, error, &fb_var);
        vd->resX = fb_var.xres;
        vd->resY = fb_var.yres;
        VS_MESSAGE("MAX resolution %dx%d\n", vd->resX, vd->resY);

        close(fd);
    }

    char * palpha;

    if (palpha = getenv("VSALPHA"))
        vd->setalpha = 1;
    else
        vd->setalpha = 0;

    fd = _getDevicefd(vd);

    VS_IOCTL(fd, FBIOGET_VSCREENINFO, error,&fb_var);

    vd->fbvar = fb_var;
    
error:
    return;
}


void
_fillDeviceLocalAlphaBuf(VideoDevice * vd, char * lbuf0, char * lbuf1)
{
    VideoSurface * vs = DEVICE2HEADSURFACE(vd);
    int stride = vd->disp.right-vd->disp.left;
    while(vs){
        int xoff, yoff;
        int width, height;
        int i;
        char * bufp0, * bufp1;
        xoff = vs->adjustdesrect.left-vd->disp.left;
        yoff = vs->adjustdesrect.top-vd->disp.top;
        width = vs->adjustdesrect.right-vs->adjustdesrect.left;
        height= vs->adjustdesrect.bottom-vs->adjustdesrect.top;
        bufp0 = lbuf0+stride*yoff+xoff;
        bufp1 = lbuf1+stride*yoff+xoff;
        for (i=0;i<height;i++){
            memset(bufp0, ALPHA_SOLID, width);
            bufp0+=stride;
            memset(bufp1, ALPHA_SOLID, width);
            bufp1+=stride;
        }
        vs = NEXTSURFACE(vs);
    };
}


int
_setAlpha(VideoDevice * vd)
{

    int fd;
    unsigned long alpha_buf0;
    unsigned long alpha_buf1;
    unsigned long loc_alpha_phy_addr0;
    unsigned long loc_alpha_phy_addr1;
    
    struct mxcfb_loc_alpha l_alpha;

    unsigned long l_alpha_buf_size;


    //return 0;

    fd = _getDevicefd(vd);

    if (0){//(vd->cnt==1){
        struct mxcfb_gbl_alpha g_alpha;
        printf("set global alpha\n");
        g_alpha.alpha = ALPHA_SOLID;
        g_alpha.enable = 1;
        VS_IOCTL(fd, MXCFB_SET_GBL_ALPHA, done, &g_alpha);
    }else{
    
    l_alpha.enable = 1;
    l_alpha.alpha_in_pixel = 0;
    l_alpha.alpha_phy_addr0 = 0;
    l_alpha.alpha_phy_addr1 = 0;

    
    VS_IOCTL(fd, MXCFB_SET_LOC_ALPHA, done, &l_alpha);

    VS_MESSAGE("get phy0 %x, phy1 %x, enable %d pixel %d\n", l_alpha.alpha_phy_addr0, l_alpha.alpha_phy_addr1,
        l_alpha.enable, l_alpha.alpha_in_pixel);

    l_alpha_buf_size = (vd->disp.right-vd->disp.left)*(vd->disp.bottom-vd->disp.top);

    loc_alpha_phy_addr0 = (unsigned long)(l_alpha.alpha_phy_addr0);
    loc_alpha_phy_addr1 = (unsigned long)(l_alpha.alpha_phy_addr1);
    
    alpha_buf0 = (char *)mmap(0, l_alpha_buf_size,
                 PROT_READ | PROT_WRITE,
                 MAP_SHARED, fd,
                 loc_alpha_phy_addr0);
    if ((int)alpha_buf0 == -1) {
        VS_ERROR("Error: failed to map alpha buffer 0"
               " to memory.\n");
        goto done;
    }
    alpha_buf1 = (char *)mmap(0, l_alpha_buf_size,
                 PROT_READ | PROT_WRITE,
                 MAP_SHARED, fd,
                 loc_alpha_phy_addr1);
    if ((int)alpha_buf1 == -1) {
        VS_ERROR("Error: failed to map alpha buffer 1"
               " to memory.\n");
        munmap((void *)alpha_buf0, l_alpha_buf_size);
        return -1;
    }

    memset(alpha_buf0, ALPHA_TRANSPARENT, l_alpha_buf_size);
    memset(alpha_buf1, ALPHA_TRANSPARENT, l_alpha_buf_size);

    _fillDeviceLocalAlphaBuf(vd, alpha_buf0, alpha_buf1);
    munmap(alpha_buf0, l_alpha_buf_size);
    munmap(alpha_buf1, l_alpha_buf_size);
    }
done:   
    return 0;
}



int
_setDeviceConfig(VideoDevice * vd)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);
    
    struct fb_var_screeninfo fb_var;
    struct fb_fix_screeninfo fb_fix;
    struct mxcfb_pos pos;
    Rect * rect;
    
    int fd = _getDevicefd(vd);

    /* Workaround for ipu hardware, it need set to 0,0 before change another offset */    
    pos.x = 0;
    pos.y = 0;
    VS_IOCTL(fd, MXCFB_SET_OVERLAY_POS, done, &pos);

    VS_IOCTL(fd, FBIOBLANK, done, FB_BLANK_POWERDOWN);

    VS_MESSAGE("Set device win"WIN_FMT"\n", WIN_ARGS(&vd->disp));

    rect = &vd->disp;
    
    VS_IOCTL(fd, FBIOGET_VSCREENINFO, done,&fb_var);

    fb_var.xres = RECT_WIDTH(rect);
    fb_var.xres_virtual = fb_var.xres;
    fb_var.yres = RECT_HEIGHT(rect);
    fb_var.yres_virtual = fb_var.yres * 2;
    fb_var.activate |= FB_ACTIVATE_FORCE;
    fb_var.nonstd = gen_fourcc('R','G','B','P');
    
    vd->fmt = gen_fourcc('R','G','B','P');
    fb_var.bits_per_pixel = 16;
    VS_IOCTL(fd, FBIOPUT_VSCREENINFO, done,&fb_var);

    VS_IOCTL(fd, FBIOGET_VSCREENINFO, done, &fb_var);
    VS_IOCTL(fd, FBIOGET_FSCREENINFO, done, &fb_fix);
    
    pos.x = vd->disp.left;
    pos.y = vd->disp.top;
    VS_IOCTL(fd, MXCFB_SET_OVERLAY_POS, done, &pos);

    VS_IOCTL(fd, FBIOBLANK, done, FB_BLANK_UNBLANK);
    vd->bufaddr[0] = fb_fix.smem_start;
    vd->bufaddr[1] = fb_fix.smem_start + fb_var.yres * fb_fix.line_length;

#ifdef METHOD2
    _initVDIPUTask(vd);
#endif

done:   
    return 0;
}


int
_closeDevice(VideoDevice * vd)
{
    int fd = _getDevicefd(vd);

    VS_IOCTL(fd, FBIOPUT_VSCREENINFO, done, &vd->fbvar);
    VS_IOCTL(fd, FBIOBLANK, done, FB_BLANK_POWERDOWN);
done:    
    return 0;
}

int
_openDevice(VideoDevice * vd)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);
    int fd = _getDevicefd(vd);
    
    return 0;
    
error:
    return -1;
}


int
_needRender(VideoSurface * curSurf, Updated * updated, int renderidx)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    if (curSurf->paddr==NULL)
        return 0;
    if (curSurf->status==VS_STATUS_INVISIBLE)
        return 0;
#ifndef METHOD2    
    if ((curSurf->rendmask & (1<<renderidx))==0)
        return 1;
#else    
    if (curSurf->rendmask==0)
        return 1;
#endif
    if ((updated->updated) && (OVERLAPED_RECT((&(curSurf->adjustdesrect)), (&updated->rect))))
        return 1;

    return 0;
}

int 
_renderSuface(VideoSurface * surf, VideoDevice * vd, Updated * updated)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    Rect * surfrect = &(surf->adjustdesrect);
    
    if (updated->updated==0){
        updated->updated = 1;
        updated->rect = *surfrect;
    }else{
        Rect * updatedfileld = &updated->rect;
        if (surfrect->left<updatedfileld->left)
            updatedfileld->left = surfrect->left;
        if (surfrect->right>updatedfileld->right)
            updatedfileld->right = surfrect->right;
        if (surfrect->top<updatedfileld->top)
            updatedfileld->top = surfrect->top;
        if (surfrect->bottom>updatedfileld->bottom)
            updatedfileld->bottom = surfrect->bottom;
    }

    surf->itask.input.user_def_paddr[0] = surf->paddr;
    surf->itask.output.user_def_paddr[0] = vd->bufaddr[vd->renderidx];

    KICK_IPUTASKONE(&surf->itask);
    
    surf->rendmask|=(1<<vd->renderidx);
    return 0;
}

int 
_FlipOnDevice(VideoDevice * vd)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

	struct fb_var_screeninfo fb_var;
    int ret;
    int fd = _getDevicefd(vd);

    VS_IOCTL(fd, FBIOGET_VSCREENINFO, done, &fb_var);

	if (vd->renderidx == 0)
		fb_var.yoffset = 0;
	else
		fb_var.yoffset = fb_var.yres;

    VS_FLOW("render  %d %d\n", vd->renderidx, fb_var.yoffset);

    VS_IOCTL(fd, FBIOPAN_DISPLAY, done, &fb_var);
    
    vd->renderidx= NEXT_RENDER_ID(vd->renderidx);
    
    VS_FLOW("render  %dfinish\n", vd->renderidx);
done:
    return 0;
}




