/*
 * Copyright (C) 2005-2010 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_mp4demuxer.c
 *
 * Description:    mp4 Demuxer plug-in for Gstreamer.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 */

/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <string.h>
#include <gst/gst.h>
#include <memory.h>
#include "mfw_gst_mp4demuxer.h"

//#define MEMORY_DEBUG

#include "mfw_gst_utils.h"



/*=============================================================================
                           STATIC VARIABLES
=============================================================================*/
static GstStaticPadTemplate mfw_gst_mp4demuxer_sink_template_factory
    = GST_STATIC_PAD_TEMPLATE("sink",
			      GST_PAD_SINK,
			      GST_PAD_ALWAYS,
	              GST_STATIC_CAPS("video/quicktime;"
	              "audio/x-m4a;""application/x-3gp")
    );

/*=============================================================================
                            LOCAL MACROS
=============================================================================*/

#define GST_CAT_DEFAULT mfw_gst_mp4demuxer_debug
/* each time, data in size */
#define MP4_QUERY_SIZE          8192
/* define the sleep time of thread, in order not to crash of demuxer and codec */
#define MP4_DEMUX_WAIT_INTERVAL 5000
#define GST_BUFFER_FLAG_IS_SYNC (GST_BUFFER_FLAG_LAST<<2)


/* define a tag name */
#define MFW_GST_TAG_WIDTH               "width"
#define MFW_GST_TAG_HEIGHT              "height"
#define MFW_GST_TAG_FRAMERATE           "framerate"
#define MFW_GST_TAG_SAMPLING_FREQUENCY  "sampling frequency"
#define MFW_GST_TAG_YEAR                "year"



enum {
    ID_0,
    ID_DECODE_AUDIO_TRACK,
    ID_DECODE_VIDEO_TRACK
};




/*=============================================================================
                           LOCAL VARIABLES
=============================================================================*/
/* None. */

/*=============================================================================
                        STATIC FUNCTION PROTOTYPES
=============================================================================*/
GST_DEBUG_CATEGORY_STATIC(mfw_gst_mp4demuxer_debug);
static void mfw_gst_mp4_demuxer_class_init(MFW_GST_MP4DEMUX_INFO_CLASS_T
                       *);
static void mfw_gst_mp4demuxer_base_init(MFW_GST_MP4DEMUX_INFO_CLASS_T *);
static void mfw_gst_mp4_demuxer_init(MFW_GST_MP4DEMUX_INFO_T *,
                     MFW_GST_MP4DEMUX_INFO_CLASS_T *);
static void mfw_gst_mp4demuxer_taskfunc(GstPad    *src_pad);
static void MP4CreateTagList(MFW_GST_MP4DEMUX_INFO_T *);                
static gboolean mfw_gst_mp4demuxer_seek(MFW_GST_MP4DEMUX_INFO_T*,GstEvent*);
static void mfw_gst_mp4_demuxer_finalize(GObject * object);
static void mfw_gst_mp4_demuxer_close(MFW_GST_MP4DEMUX_INFO_T *);
static void mfw_gst_mp4demuxer_set_property(GObject * object, guint prop_id,
			    const GValue * value, GParamSpec * pspec);
static void mfw_gst_mp4demuxer_get_property(GObject * object, guint prop_id,
			    GValue * value, GParamSpec * pspec);



/*=============================================================================
                            STATIC VARIABLES
=============================================================================*/


static GstElementClass *parent_class = NULL;

/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/

/*=============================================================================
FUNCTION: mfw_gst_mp4demuxer_set_property 

DESCRIPTION: sets the property of the mp4 element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property set by the application
        pspec      - pointer to the attributes of the property

RETURN VALUE:
        None
=============================================================================*/

static void
mfw_gst_mp4demuxer_set_property(GObject * object, guint prop_id,
			    const GValue * value, GParamSpec * pspec)
{
    MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer = MFW_GST_MP4_DEMUXER(object);
    switch (prop_id) 
    {
        case ID_DECODE_AUDIO_TRACK:
            mp4_demuxer->audio_track_decode = g_value_get_int(value);
            GST_DEBUG("selected audio track to decode is: %d\n", mp4_demuxer->audio_track_decode);
            break;

        case ID_DECODE_VIDEO_TRACK:
            mp4_demuxer->video_track_decode = g_value_get_int(value);
            GST_DEBUG("selected video track to decode is: %d\n", mp4_demuxer->video_track_decode);
            break;			

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
            break;
    }
}


/*=============================================================================
FUNCTION: mfw_gst_mp4demuxer_get_property

DESCRIPTION: gets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property to be set for the next element
        pspec      - pointer to the attributes of the property

RETURN VALUE:
        None
=============================================================================*/
static void
mfw_gst_mp4demuxer_get_property(GObject * object, guint prop_id,
			    GValue * value, GParamSpec * pspec)
{

    MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer = MFW_GST_MP4_DEMUXER(object);

    switch (prop_id) 
    {
        case ID_DECODE_AUDIO_TRACK:
            GST_DEBUG("current audio track to decode is: %d\n", mp4_demuxer->audio_track_decode);
			g_value_set_int(value, mp4_demuxer->audio_track_decode);
            break;

        case ID_DECODE_VIDEO_TRACK:
            GST_DEBUG("current video track to decode is: %d\n", mp4_demuxer->video_track_decode);
			g_value_set_int(value, mp4_demuxer->video_track_decode);
            break;			
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
            break;
    }
} 

/*=============================================================================
FUNCTION:			 audio_src_templ
		
DESCRIPTION:		 defines the audio source pad template.

ARGUMENTS PASSED:	 None
		
  
RETURN VALUE:		 Pointer to GstPadTemplate. 
=============================================================================*/

static GstPadTemplate *audio_src_templ(void)
{
	static GstPadTemplate *templ = NULL;

	if (!templ) {
	GstCaps *caps = NULL;
	GstCaps *caps_PCM, *caps_amr_wb, *caps_amr_nb; 

	caps_PCM = NULL;
	caps_amr_wb = NULL;
	caps_amr_nb = NULL;
	
	caps = gst_caps_new_simple("audio/mpeg",
				   "mpegversion", GST_TYPE_INT_RANGE, 1, 4,
				   "bitrate", GST_TYPE_INT_RANGE, 0,
				   G_MAXINT,NULL);

	caps_PCM = gst_caps_new_simple("audio/x-raw-int",
				   "rate", GST_TYPE_INT_RANGE, 1000,96000,
				   "channels",GST_TYPE_INT_RANGE,1,2,
				   "width", GST_TYPE_INT_RANGE, 8, 32,
				   "depth", GST_TYPE_INT_RANGE, 1, 32,
	               "endianness", GST_TYPE_INT_RANGE, G_LITTLE_ENDIAN,G_BIG_ENDIAN,
	               //"signed", G_TYPE_BOOLEAN, {TRUE, FALSE},
				   NULL);

    caps_amr_wb = gst_caps_new_simple("audio/AMR-WB",
                   "bitrate", GST_TYPE_INT_RANGE, 0,G_MAXINT,
                   "rate", GST_TYPE_INT_RANGE, 8000,16000,
                   "channels",GST_TYPE_INT_RANGE,1,2,
                   NULL);

    caps_amr_nb = gst_caps_new_simple("audio/AMR",
                   "bitrate", GST_TYPE_INT_RANGE, 0,G_MAXINT,
                   "rate", GST_TYPE_INT_RANGE, 8000,16000,
                   "channels",GST_TYPE_INT_RANGE,1,2,
                   NULL);

	


	gst_caps_append(caps,caps_PCM);
	gst_caps_append(caps,caps_amr_wb);
	gst_caps_append(caps,caps_amr_nb);

	templ = gst_pad_template_new(PARSER_AUDIOPAD_TEMPLATE_NAME, GST_PAD_SRC,
					 GST_PAD_SOMETIMES, caps);
	}
	return templ;
}

/*=============================================================================
FUNCTION:			 video_src_templ
		
DESCRIPTION:		 defines the audio source pad template.

ARGUMENTS PASSED:	 None
		
  
RETURN VALUE:		 Pointer to GstPadTemplate. 
=============================================================================*/

static GstPadTemplate *video_src_templ()
{
	static GstPadTemplate *templ = NULL;

	if (!templ) {
	GstCaps *caps	   = NULL;
	GstCaps *caps_h264 = NULL;
	GstCaps *caps_h263 = NULL;	
	GstCaps *caps_mjpeg = NULL;

	caps	 = gst_caps_new_simple("video/mpeg",
				   "mpegversion", GST_TYPE_INT_RANGE, 1, 4,
				   "systemstream", GST_TYPE_INT_RANGE, 0, 1,
				   "width", GST_TYPE_INT_RANGE, MIN_RESOLUTION_WIDTH, MAX_RESOLUTION_WIDTH,
				   "height", GST_TYPE_INT_RANGE, MIN_RESOLUTION_HEIGHT, MAX_RESOLUTION_HEIGHT,
				   NULL);


	caps_h264 = gst_caps_new_simple("video/x-h264",
				   "systemstream", GST_TYPE_INT_RANGE, 0, 1, 
				   "width", GST_TYPE_INT_RANGE, MIN_RESOLUTION_WIDTH, MAX_RESOLUTION_WIDTH,
				   "height", GST_TYPE_INT_RANGE, MIN_RESOLUTION_HEIGHT, MAX_RESOLUTION_HEIGHT,
				   NULL);
	caps_h263 = gst_caps_new_simple("video/x-h263",
				   "systemstream", GST_TYPE_INT_RANGE, 0, 1, 
				   "width", GST_TYPE_INT_RANGE, MIN_RESOLUTION_WIDTH, MAX_RESOLUTION_WIDTH,
				   "height", GST_TYPE_INT_RANGE, MIN_RESOLUTION_HEIGHT, MAX_RESOLUTION_HEIGHT,
				   NULL);
	

	caps_mjpeg = gst_caps_new_simple("image/jpeg",
				   "systemstream", GST_TYPE_INT_RANGE, 0, 1, 
				   "width", GST_TYPE_INT_RANGE, MIN_RESOLUTION_WIDTH, MAX_RESOLUTION_WIDTH,
				   "height", GST_TYPE_INT_RANGE, MIN_RESOLUTION_HEIGHT, MAX_RESOLUTION_HEIGHT,
				   NULL);

	
	
	gst_caps_append(caps,caps_h264);
	gst_caps_append(caps,caps_h263);
	gst_caps_append(caps,caps_mjpeg);

	templ = gst_pad_template_new(PARSER_VIDEOPAD_TEMPLATE_NAME, GST_PAD_SRC,
					 GST_PAD_SOMETIMES, caps);
	}
	return templ;
}


 /*=============================================================================
 FUNCTION:			  MP4LocalCalloc
		 
 DESCRIPTION:		  Implements the calloc function
   
 
 ARGUMENTS PASSED:
		 TotalNumber -	 total number of memory blocks
		 TotalSize	 -	 size of each block
		 
   
 RETURN VALUE:		   memory pointer
 
 =============================================================================*/
 
 static void *app_MP4LocalCalloc(guint32 TotalNumber, guint32 TotalSize)
 {
 
 
	 /* Void Pointer. */
	 void *PtrCalloc = NULL;
	 if (TotalNumber==0) TotalNumber=1;
	 /* Allocate the memory. */
	 PtrCalloc = (void *) MM_MALLOC(TotalNumber * TotalSize);

	 if(PtrCalloc != NULL)
	     memset(PtrCalloc, 0, TotalNumber * TotalSize);
         else
	 	GST_ERROR("app_MP4LocalCalloc failed, return zero. \n");
	 return (PtrCalloc);
 }
 
 /*=============================================================================
 FUNCTION:			  MP4LocalMalloc
		 
 DESCRIPTION:		  Implements the malloc function
   
 
 ARGUMENTS PASSED:
		 TotalSize	 - size of the memory requested
		 
   
 RETURN VALUE:		   memory pointer
		 
 =============================================================================*/
 
 static void *app_MP4LocalMalloc(guint32 TotalSize)
 {
 
	 /* Void pointer to the malloc. */
	 void *PtrMalloc = NULL;
 
	 /* Allocate the mempry. */
	 PtrMalloc = MM_MALLOC(TotalSize);
	 if(PtrMalloc != NULL)
	     memset(PtrMalloc, 0, TotalSize);
     else
	 	GST_ERROR("app_MP4LocalMalloc failed, return zero. \n");
	 
	 return (PtrMalloc);
 }
 
 /*=============================================================================
 FUNCTION:			  app_MP4LocalReAlloc
		 
 DESCRIPTION:		  Implements the memory reallocation function
   
 
 ARGUMENTS PASSED:
		 MemoryBlock - memory pointer
		 
   
 RETURN VALUE:		   memory pointer
		 
 
 PRE-CONDITIONS:
		 None
  
 POST-CONDITIONS:
		 None
 
 IMPORTANT NOTES:
		 None
 =============================================================================*/
 
 static void *app_MP4LocalReAlloc(void *MemoryBlock, guint32 TotalSize)
 {
 
	 /* Void Pointer. */
	 void *PtrRealloc = NULL;
 
	 /* Re alloc the memory. */
	 PtrRealloc = (void *)g_realloc(MemoryBlock, TotalSize);
     if(PtrRealloc == NULL)
	 	GST_ERROR("app_MP4LocalReAlloc failed, return zero. \n");
	 return (PtrRealloc);
 }
 
 
 /*=============================================================================
 FUNCTION:			  MP4LocalFree
		 
 DESCRIPTION:		  Implements the memory free function
   
 
 ARGUMENTS PASSED:
		 MemoryBlock - memory pointer
		 
   
 RETURN VALUE:		   memory pointer
 
 =============================================================================*/
 
 static void app_MP4LocalFree(void *MemoryBlock)
 {
	 MM_FREE(MemoryBlock);
 }
 
 
 /*=============================================================================
 FUNCTION:			  MP4PullData
		 
 DESCRIPTION:		  Implements pulling a buffer from peer pad
   
 
 ARGUMENTS PASSED:
		 moved_loc -  The start offset of the buffer 
		 
   
 RETURN VALUE:		  GstBuffer pointer
 
 =============================================================================*/
 
 static GstBuffer *MP4PullData(MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer)
 {
	 guint64 moved_loc=mp4_demuxer->file_info.buf_offset;
	 GstBuffer *tmpbuf = NULL;
 
	 /* Pulls a buffer from the peer pad */
	 if (gst_pad_pull_range(mp4_demuxer->sinkpad, moved_loc,
				MP4_QUERY_SIZE, &tmpbuf) != GST_FLOW_OK) 
	 {
	     GST_ERROR(" FILE_READ: not able to read the data from %lld\n",
		   moved_loc);
	     return NULL;
 
	 } 
	 else
	     return tmpbuf;
 
 }
 
 
 /*=============================================================================
 FUNCTION:			  MP4LocalReadFile
		 
 DESCRIPTION:		  This function reads the requestd amount of data from
					  a block of data.
  
   
 ARGUMENTS PASSED:		  
	   SourceBuffer   buffer to write the data 
	   TotalSize	  size of each of block of data
	   DestStream	  the file handle.
 
 
 RETURN VALUE : 	  size of the data read.
		 
 =============================================================================*/
 
 static guint32 app_MP4LocalReadFile(file_stream_t *stream, void *SourceBuffer, guint32 TotalSize, void *app_context)
 {
 
	 GstBuffer *residue_buf = NULL;
	 GstBuffer *new_buf = NULL;
	 MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer = NULL;
	 guint32 residue_size = 0;
	 guint32 data_req = 0;
	 mp4_demuxer = (MFW_GST_MP4DEMUX_INFO_T *)app_context;
 
	 data_req = TotalSize;	 /* requested data size */
 
	 while (1)
	 {
	 residue_size = mp4_demuxer->buf_size;
	 if (data_req > residue_size) {
 
		 if (residue_size != 0) {	 /* if data is remaining in the buffer */
 
		 residue_buf = gst_buffer_new_and_alloc(residue_size);
 
		 memcpy(residue_buf->data, mp4_demuxer->inbuff,
				residue_size);
 
		 /* pulls block of data */
		 new_buf = MP4PullData(mp4_demuxer);
 
		 if (new_buf == NULL) {
			 GST_ERROR("not able to pull the data\n");
			 return 0;
		 }
		 if (mp4_demuxer->tmpbuf != NULL) {
			 gst_buffer_unref(mp4_demuxer->tmpbuf);
			 mp4_demuxer->tmpbuf = NULL;
		 }
		 mp4_demuxer->tmpbuf =
			 gst_buffer_join(residue_buf, new_buf);
		 } else {
		 /* if no data is remaining in the buffer */
 
		 if (mp4_demuxer->tmpbuf != NULL) {
			 gst_buffer_unref(mp4_demuxer->tmpbuf);
			 mp4_demuxer->tmpbuf = NULL;
		 }
 
		 /* pulls block of data */
		 mp4_demuxer->tmpbuf =
			 MP4PullData(mp4_demuxer);
 
		 if (mp4_demuxer->tmpbuf == NULL) {
			 GST_ERROR("not able to pull the data\n");
			 return 0;
		 }
 
		 }
 
		 mp4_demuxer->buf_size = GST_BUFFER_SIZE(mp4_demuxer->tmpbuf);	 /* buffer size */
		 mp4_demuxer->inbuff = GST_BUFFER_DATA(mp4_demuxer->tmpbuf); /* buffer pointer */
		 mp4_demuxer->file_info.buf_offset += MP4_QUERY_SIZE;	 /* next buffer offset */
 
		 if (data_req <= mp4_demuxer->buf_size)
		 break;
		 else
		 continue;
 
	 }
	 break;
	 }
 
	 memcpy(SourceBuffer, mp4_demuxer->inbuff, data_req);
 
	 mp4_demuxer->file_info.offset += data_req;
 
	 mp4_demuxer->inbuff += data_req;
	 mp4_demuxer->buf_size -= data_req;
 
	 return (data_req);
 
 }
 
 
 /*=============================================================================
 FUNCTION:			  MP4LocalFileSize
		 
 DESCRIPTION:		  This function gets the size of the file.
 
 ARGUMENTS PASSED:	  None
		 
   
 RETURN VALUE:		  the size of the file.
		 
 =============================================================================*/
 
 static gint64 app_MP4LocalFileSize(file_stream_t *stream, void *app_context)
 {
	 GstPad *my_peer_pad = NULL;
	 MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer = NULL;
	 GstFormat fmt = GST_FORMAT_BYTES;
 
	 mp4_demuxer = (MFW_GST_MP4DEMUX_INFO_T *)app_context;
 
	 my_peer_pad = gst_pad_get_peer(mp4_demuxer->sinkpad);
	 gst_pad_query_duration(my_peer_pad, &fmt,
				&(mp4_demuxer->file_info.length));
 
	 gst_object_unref(my_peer_pad);
	 return (mp4_demuxer->file_info.length);
 }
 
 /*=============================================================================
 FUNCTION:			  MP4LocalGetCurrentFilePos
		 
 DESCRIPTION:		  this function gets the current position of the file handler
 
 ARGUMENTS PASSED:	  
		 fileHandle   the handle to the file.
   
 RETURN VALUE:		  the position of the file handler
 
 =============================================================================*/
 
 static gint64 app_MP4LocalGetCurrentFilePos(file_stream_t *stream, void *app_context)
 {
	 MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer = NULL;
	 mp4_demuxer = (MFW_GST_MP4DEMUX_INFO_T *)app_context;
 
	 return (mp4_demuxer->file_info.offset);
 }
 
 /*=============================================================================
 FUNCTION:			  MP4LocalFileOpen
		 
 DESCRIPTION:		  this function gets the file handle of the file.
 
 ARGUMENTS PASSED:	  
	   FileName 	  the name of the file
	   ModeToOpen	  mode in which the file has to be opened.
			   
   
 RETURN VALUE:		  NULL pointer
		 
 =============================================================================*/
 
 static gint app_MP4LocalFileOpen(file_stream_t *stream, const uint8 * mode, void * context)
 {
	 return 0;
 }
 
 /*=============================================================================
 FUNCTION:			  MP4LocalFileClose
		 
 DESCRIPTION:		  this function gets the file handle of the file.
 
 ARGUMENTS PASSED:	  
	   FileName 	  the name of the file
	   ModeToOpen	  mode in which the file has to be opened.
			   
   
 RETURN VALUE:		  NULL pointer
		 
 =============================================================================*/
 
 static gint app_MP4LocalFileClose(file_stream_t *stream, void * context)
 {
	 return 0;
 }
 
 
 /*=============================================================================
 FUNCTION:			  MP4LocalSeekFile
		 
 DESCRIPTION:		  this function moves the file handle to the desired number of locations
 
 ARGUMENTS PASSED:	  
	   fileHandle	  the handle to the file
	   offset		  offset to which the handler to be moved.
	   origin		  the file handler position from which it has to be moved
		 
 =============================================================================*/
 
 static gint64 app_MP4LocalSeekFile(file_stream_t *stream, gint64 offset, gint origin,
				  void *app_context)			
 {
	 int seekStatus = 0;
	 GstBuffer *tmpbuf = NULL;
	 int file_read = 0;
	 int index = 0;
	 guint64 moved_loc = 0;
	 MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer = NULL;
 
	 mp4_demuxer = (MFW_GST_MP4DEMUX_INFO_T *)app_context;
 
	 moved_loc = mp4_demuxer->file_info.offset;
 
	 GST_DEBUG("file seek with offset=%lld and origin = %d\n",
		   offset, origin);
 
	 switch (origin) {
	 case SEEK_SET:
	 {
		 moved_loc = 0;
		 moved_loc += offset;
		 break;
	 }
 
	 case SEEK_CUR:
	 {
		 moved_loc += offset;
		 break;
	 }
 
	 case SEEK_END:
	 {
		 moved_loc = mp4_demuxer->file_info.length;
		 break;
	 }
 
	 default:
	 {
 
		 break;
	 }
	 }
 
	 mp4_demuxer->file_info.offset = moved_loc;
	 mp4_demuxer->file_info.buf_offset = moved_loc;
	 mp4_demuxer->buf_size = 0;
	 return moved_loc;
 }



/*=============================================================================
FUNCTION:   	mfw_gst_mp4demuxer_parser_init
	
DESCRIPTION:	
            This function initialize the parser .

ARGUMENTS PASSED:	
            demuxer 	- pointer to the mp4 demuxer element.
				
	 
RETURN VALUE:
		    GST_STATE_CHANGE_SUCCESS    -   success.
            GST_STATE_CHANGE_FAILURE    -   failure.

=============================================================================*/
static MP4_PARSER_ERROR_CODE
mfw_gst_mp4demuxer_parser_init(MFW_GST_MP4DEMUX_INFO_T * demuxer)
{

    gint uliResult;
	const char *name = "GST_ACCURATESEEK";
    gint i = 0;
  
    /* all var need to be cleared, and then if necessary, assgn init value to specify var */
	/***** Initializing the Demuxer new added elements. *****/

    /* below var is add for new core parser */
	demuxer->file_info.buf_offset = 0;
	demuxer->file_info.length = 0;
	demuxer->file_info.offset = 0;
    demuxer->tmpbuf = NULL;
    demuxer->buf_size = 0;
    demuxer->inbuff = NULL;
    demuxer->stream.file_handle = NULL;
	demuxer->stream.file_path = NULL;
    demuxer->parserHandle = NULL;
    demuxer->context = NULL;
	demuxer->segment.start = 0;
	demuxer->total_src_pad = 0;

	memset(&(demuxer->tracks[0]), 0, MAX_TRACK_NUM*sizeof(Track));
	memset(&(demuxer->usrdata[0]), 0, MP4_USRDATA_NUM*sizeof(MP4_UsrData));
	memset(&(demuxer->reorder_track), 0, sizeof(TrackReorder));
	
    demuxer->stream.pf_fopen = app_MP4LocalFileOpen;
    demuxer->stream.pf_fread = app_MP4LocalReadFile;
    demuxer->stream.pf_fseek = app_MP4LocalSeekFile;
    demuxer->stream.pf_ftell = app_MP4LocalGetCurrentFilePos;
    demuxer->stream.pf_fsize = app_MP4LocalFileSize;
    demuxer->stream.pf_fclose = app_MP4LocalFileClose;

    demuxer->memOps.Calloc = app_MP4LocalCalloc;
    demuxer->memOps.Malloc = app_MP4LocalMalloc;
    demuxer->memOps.Free = app_MP4LocalFree;
	demuxer->memOps.ReAlloc=   app_MP4LocalReAlloc;
    demuxer->context = (void *)demuxer;
    demuxer->accurate_seek = NULL;

    /* get envionment var for accurate seek */
    demuxer->accurate_seek = getenv(name);
    if(demuxer->accurate_seek != NULL)
    {
	    GST_DEBUG("mp4 accurate seek value is: %c\n", *(demuxer->accurate_seek));
    }
	
	//g_print("\n, enter CREAT parser module\n");
	uliResult = MP4CreateParser(&(demuxer->stream),
								&(demuxer->memOps),
								demuxer->context,
								&(demuxer->parserHandle));

	//g_print("\n, OUT FROM parser module%d\n", uliResult);
    GST_DEBUG(" uliResult = %d\n", uliResult);
	if(uliResult != PARSER_SUCCESS)
	{
		GST_ERROR("\nFATAL ERROR during parser create initialization\n");
		return uliResult;
	}
	
    return PARSER_SUCCESS;

}
/*=============================================================================
FUNCTION:            mp4_demuxer_track_reorder
        
DESCRIPTION:         reorder all the tracks, record each track info into array.
                     

ARGUMENTS PASSED:    demuxer   The main structure of mp4 parser plugin context.
        
  
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd
        

=============================================================================*/
static gboolean mp4_demuxer_track_reorder(MFW_GST_MP4DEMUX_INFO_T *demuxer)
{
	gint i, audio_index, video_index;
	MP4_PARSER_ERROR_CODE err = PARSER_SUCCESS;
    audio_index = 0;
	video_index = 0;
	for(i = 0; i < demuxer->total_tracks; i++)
	{
        err = MP4GetTrackType(demuxer->parserHandle,
                              i, 
                              &(demuxer->tracks[i].mediaType),
                              &(demuxer->tracks[i].decoderType),
                              &(demuxer->tracks[i].decoderSubtype));		
		if(err != PARSER_SUCCESS)
			return FALSE;
		if(demuxer->tracks[i].mediaType == MEDIA_AUDIO)
			demuxer->reorder_track.audiotrack[audio_index++] = i;
		if(demuxer->tracks[i].mediaType == MEDIA_VIDEO)
			demuxer->reorder_track.videotrack[video_index++] = i;
	}
	demuxer->reorder_track.audio_track_num = audio_index;
	demuxer->reorder_track.video_track_num = video_index;
	//g_print("total audio track is: %d, video is: %d\n", audio_index, video_index);
	return TRUE;
}
/*=============================================================================
FUNCTION:            mp4_demuxer_find_audio_track
        
DESCRIPTION:         to see whether usr choose some audio track, if not
                     select one for each to play, if choose, then try to verify 
                     whether it is valid, if valid, assgin it, if not, search one for it.
                     if can not find one type, use -1 to make it

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
        
  
RETURN VALUE:
none
        

=============================================================================*/
static void mp4_demuxer_find_audio_track(MFW_GST_MP4DEMUX_INFO_T *demuxer)
{
    gint i, track, audio_find;

	//g_print(" selected audio track is: %d\n", demuxer->audio_track_decode);

	/* error handle for track select */
    if(demuxer->audio_track_decode < 0 || demuxer->audio_track_decode > demuxer->total_tracks)
		demuxer->audio_track_decode = 0;
	/* audio track select */
	track = demuxer->audio_track_decode;
	audio_find = 0;
	if(track != 0)
	{   /* search all audio tracks, find it out */
        for(i = 0; i < demuxer->reorder_track.audio_track_num; i++)
        {
            if(track == demuxer->reorder_track.audiotrack[i])
            {
                audio_find = 1;
				break;
			}
		}
		if(audio_find)
			demuxer->audio_track_decode = track;	
		else
		{
			if(demuxer->reorder_track.audio_track_num >= 1)
			    demuxer->audio_track_decode = demuxer->reorder_track.audiotrack[0];
		    else
			    /* -1 indicates no audio track found */
			    demuxer->audio_track_decode = -1;
		}
	}
	else
	{
		if(demuxer->reorder_track.audio_track_num >= 1)
			demuxer->audio_track_decode = demuxer->reorder_track.audiotrack[0];
		else
			/* -1 indicates no audio track found */
			demuxer->audio_track_decode = -1;  
	}
	//g_print("find audio track: %d to decode\n", demuxer->audio_track_decode);
}
/*=============================================================================
FUNCTION:            mp4_demuxer_find_video_track
        
DESCRIPTION:         to see whether usr choose some video track, if not
                     select one for each to play, if choose, then try to verify 
                     whether it is valid, if valid, assgin it, if not, search one for it.
                     if can not find one type, use -1 to make it

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
        
  
RETURN VALUE:
none
        

=============================================================================*/
static void mp4_demuxer_find_video_track(MFW_GST_MP4DEMUX_INFO_T *demuxer)
{
    gint i, track, video_find;

	/* error handle for track select */
    if(demuxer->video_track_decode < 0 || demuxer->video_track_decode > demuxer->total_tracks)
		demuxer->video_track_decode = 0;
	/* video track select */
	track = demuxer->video_track_decode;
	video_find = 0;
	if(track != 0)
	{   /* search all video tracks, find it out */
        for(i = 0; i < demuxer->reorder_track.video_track_num; i++)
        {
            if(track == demuxer->reorder_track.videotrack[i])
            {
                video_find = 1;
				break;
			}
		}
		if(video_find)
			demuxer->video_track_decode = track;
		else
		{
    		if(demuxer->reorder_track.video_track_num >= 1)
    			demuxer->video_track_decode = demuxer->reorder_track.videotrack[0];
    		else
    			/* -1 indicates no video track found */
    			demuxer->video_track_decode = -1;
		}
	}
	else
	{
		if(demuxer->reorder_track.video_track_num >= 1)
			demuxer->video_track_decode = demuxer->reorder_track.videotrack[0];
		else
			/* -1 indicates no video track found */
			demuxer->video_track_decode = -1;  
	}
	//g_print("find video track: %d to decode\n", demuxer->video_track_decode);	
}
/*=============================================================================
FUNCTION:            mp4_demuxer_read_common_info
        
DESCRIPTION:         read information for audio & video common

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
        
RETURN VALUE:
none

        

=============================================================================*/
static MP4_PARSER_ERROR_CODE mp4_demuxer_read_common_info(MFW_GST_MP4DEMUX_INFO_T *demuxer, gint track_num)
{
    guint32 specificInfoSize, i;
	guint8  *specifcinfoPtr;
	MP4_PARSER_ERROR_CODE err = PARSER_SUCCESS;
    specifcinfoPtr = NULL;
	specificInfoSize = 0;

	err = MP4GetSampleDuration(demuxer->parserHandle, track_num, &(demuxer->tracks[track_num].usSampleDuration));
	if(PARSER_SUCCESS != err) 
	{
		GST_DEBUG("can not get audio sample rate\n");
		return err;
	}			
	//g_print("sample duration: %ld us\n", (demuxer->tracks[i].usSampleDuration));
	if( 0 == demuxer->tracks[track_num].usSampleDuration)
		demuxer->tracks[track_num].isCbr = TRUE;
	else if(demuxer->tracks[track_num].mediaType == MEDIA_VIDEO)
		demuxer->tracks[track_num].video_framerate = (double)((double)1000000/(double)(demuxer->tracks[track_num].usSampleDuration));
	
	//g_print("usSampleDuration is %lld\n", demuxer->tracks[i].usSampleDuration);				
	//g_print("video_framerate is %f\n", demuxer->tracks[i].video_framerate);				
	err = MP4GetBitRate(demuxer->parserHandle, track_num, &(demuxer->tracks[track_num].bitRate));
	//g_print(" bitstream's bits per sec: %d\n", demuxer->tracks[i].bitRate);
	if(PARSER_SUCCESS != err) 
	{
		GST_DEBUG("can not get bitstream bit rate\n");
		return err;
	}
	
	err = MP4GetMaxSampleSize(demuxer->parserHandle, track_num, &(demuxer->tracks[track_num].maxSampleSize));
	if(PARSER_SUCCESS != err) 
	{
		GST_DEBUG("can not get bitstream's max sample size\n");
		return err;
	}			
	//g_print("max sample size: %d bytes\n", demuxer->tracks[track_num].maxSampleSize);
	
	
	err = MP4GetLanguage(demuxer->parserHandle, track_num, (guint8 *)(&(demuxer->tracks[track_num].language)));
	if(PARSER_SUCCESS != err) 
	{
		GST_DEBUG("can not get bitstream's language\n");
		return err;
	}
	err = MP4GetDecoderSpecificInfo(demuxer->parserHandle, track_num, &specifcinfoPtr, &specificInfoSize); 
	if(PARSER_SUCCESS != err)
	{
		GST_DEBUG("can not get codec specific data for tarck: %d\n", track_num);
		return err;
	}
    
	if( (specificInfoSize-MP4_BITMAPINFOSIZE) > 0)
	{
        /* assign the size into structure */
		demuxer->tracks[track_num].decoderSpecificInfo = gst_buffer_new_and_alloc(specificInfoSize-MP4_BITMAPINFOSIZE);
		if(demuxer->tracks[track_num].decoderSpecificInfo == NULL)
		{
			GST_DEBUG("can not allocate memory for specific codec data\n");
			return PARSER_INSUFFICIENT_MEMORY;
		}
		/* replace the first start code to nal length */		
		//specifcinfoPtr[0] = (guint8)((specificInfoSize-4)>>24);
        //specifcinfoPtr[1] = (guint8)((specificInfoSize-4)>>16);		
		//specifcinfoPtr[2] = (guint8)((specificInfoSize-4)>>8);
        //specifcinfoPtr[3] = (guint8)(specificInfoSize-4);		
		
		memcpy(GST_BUFFER_DATA(demuxer->tracks[track_num].decoderSpecificInfo), (specifcinfoPtr+MP4_BITMAPINFOSIZE), specificInfoSize-MP4_BITMAPINFOSIZE);
		g_print("specific info size is: %d\n", specificInfoSize);
	}
		
	
	return err;
}


/*=============================================================================
FUNCTION:            mp4_demuxer_read_av_track
        
DESCRIPTION:         read information for the specified audio and video track

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
        
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd

        

=============================================================================*/
static MP4_PARSER_ERROR_CODE mp4_demuxer_read_av_track(MFW_GST_MP4DEMUX_INFO_T *demuxer)
{
    gint audio_track, video_track;
	MP4_PARSER_ERROR_CODE err = PARSER_SUCCESS;
	
	audio_track = demuxer->audio_track_decode;
	video_track = demuxer->video_track_decode;
	
    if(audio_track != -1)
    {
        /* get the detail audio codec type */
		demuxer->tracks[audio_track].audioCodec = demuxer->tracks[audio_track].decoderType;
        /* valid audio track, decode info for it. */    	
    	err = MP4GetAudioNumChannels(demuxer->parserHandle, audio_track, &(demuxer->tracks[audio_track].numChannels));
    	if(PARSER_SUCCESS != err) 
    	{
    		GST_DEBUG("can not get audio channel number\n");
    		return err;
    	}
		/* error handling for error channel, while channel 0, make it as mono output */
		if (demuxer->tracks[audio_track].numChannels == 0)
			(demuxer->tracks[audio_track].numChannels) = 1;
    	
    	err =  MP4GetAudioSampleRate(demuxer->parserHandle, audio_track, &(demuxer->tracks[audio_track].sampleRate));
    	if(PARSER_SUCCESS != err) 
    	{
    		GST_DEBUG("can not get audio sample rate\n");
    		return err;
    	}
		if(demuxer->tracks[audio_track].audioCodec == AUDIO_PCM)
		{
        	err =  MP4GetAudioBitsPerSample(demuxer->parserHandle, audio_track, &(demuxer->tracks[audio_track].pcm_width));
        	if(PARSER_SUCCESS != err) 
        	{
        		GST_DEBUG("can not get pcm sample width\n");
        		return err;
        	}
		}
		
		err = mp4_demuxer_read_common_info(demuxer, audio_track);

        
        
    	if(PARSER_SUCCESS != err) 
    	{
    		GST_DEBUG("can not get audio common info\n");
    		return err;
    	}		
	}
	if(video_track != -1)
	{
		/* get the detail video codec type */
		demuxer->tracks[video_track].videoCodec = demuxer->tracks[video_track].decoderType;
		err = MP4GetVideoFrameWidth(demuxer->parserHandle, video_track, &(demuxer->tracks[video_track].width)); 
		if(PARSER_SUCCESS != err) 
		{
			GST_DEBUG("can not get video width\n");
			return err;
		}
		err = MP4GetVideoFrameHeight(demuxer->parserHandle, video_track, &(demuxer->tracks[video_track].height));
		if(PARSER_SUCCESS != err) 
		{
			GST_DEBUG("can not get video width\n");
			return err;
		}
					
		err = mp4_demuxer_read_common_info(demuxer, video_track);
    	if(PARSER_SUCCESS != err) 
    	{
    		GST_DEBUG("can not get video common info\n");
    		return err;
    	}		
	}
    return err;
}



/*=============================================================================
FUNCTION:            mp4_demuxer_fill_stream_info
        
DESCRIPTION:         Add and Creates the audio pad sets the 
                     corresponding capabilities.

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
                     total_tracks - Total number of tracks present in the file. 
        
  
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd
        

=============================================================================*/

static MP4_PARSER_ERROR_CODE mp4_demuxer_fill_stream_info(MFW_GST_MP4DEMUX_INFO_T *
                         demuxer)
{
	gboolean ret;
	MP4_PARSER_ERROR_CODE err = PARSER_SUCCESS;
	ret = TRUE;

	 /***********************************************************
	 step 5:  check file & track properties
	***********************************************************/
	
	err = MP4IsSeekable(demuxer->parserHandle, &(demuxer->seekable));
	if(PARSER_SUCCESS != err)
	{
		GST_DEBUG("mp4 file is not seekable\n");
		return err;
	}
	if(demuxer->seekable)
	    g_print("File is seekable\n");
	else
	    g_print("File is NOT seekable\n");
	
	ret = mp4_demuxer_track_reorder(demuxer);
	if(ret != TRUE)
	{
		GST_ERROR("can not reorder tracks\n");
		return PARSER_ERR_UNKNOWN;
	}
    mp4_demuxer_find_audio_track(demuxer);	
	mp4_demuxer_find_video_track(demuxer);
	err = mp4_demuxer_read_av_track(demuxer);
	if(PARSER_SUCCESS != err)
	{
			GST_ERROR("error in read av tracks\n");
			return err;
	}

	err = MP4GetTheMovieDuration(demuxer->parserHandle, &(demuxer->usDuration));
	if(PARSER_SUCCESS != err)
	{
			GST_DEBUG("error for get file duration\n");
			return err;
	}
	if(demuxer->usDuration == 0)
	{
	     if(demuxer->video_track_decode != -1)
	     {
             err = MP4GetTheTrackDuration(demuxer->parserHandle, demuxer->video_track_decode, &(demuxer->usDuration));
			 g_print("video track duration is: %lld\n", demuxer->usDuration);
			 if(demuxer->usDuration == 0 || err != PARSER_SUCCESS)
			 {
				 GST_DEBUG("error for get file duration2\n");
				 return err;
			 }
	     }
	}
	
	if(demuxer->audio_track_decode != -1)
        demuxer->total_src_pad++;
	if(demuxer->video_track_decode != -1)
        demuxer->total_src_pad++;

	/* if no audio and video track to decode, return error */
	if(demuxer->total_src_pad == 0)
		return PARSER_ERR_UNKNOWN;
	return PARSER_SUCCESS;
}

 /*=============================================================================
 FUNCTION:	 mfw_gst_mp4demuxer_handle_src_query   
 
 DESCRIPTION:	 performs query on src pad.    
 
 ARGUMENTS PASSED:
		 pad	 -	 pointer to GstPad
		 query	 -	 pointer to GstQuery		
			 
 RETURN VALUE:
		 TRUE	 -	 success
		 FALSE	 -	 failure
 ==============================================================================*/
 
 static gboolean mfw_gst_mp4demuxer_handle_src_query(GstPad * pad,
							 GstQuery * query)
 {
	 GstFormat format = GST_FORMAT_TIME;
	 gint64 dest_value;
 
	 MFW_GST_MP4DEMUX_INFO_T *demuxer_info;
	 gboolean res = TRUE;
	 demuxer_info = MFW_GST_MP4_DEMUXER(gst_pad_get_parent(pad));
 
	 GST_DEBUG("handling %s query",
		   gst_query_type_get_name(GST_QUERY_TYPE(query)));
 
	 switch (GST_QUERY_TYPE(query)) {
	 case GST_QUERY_DURATION:
	 {
		 
		 /* save requested format */
		 gst_query_parse_duration(query, &format, NULL);
			 
 
		 if (format == GST_FORMAT_TIME) 
		 {
		 gst_query_set_duration(query, GST_FORMAT_TIME,
						demuxer_info->segment.duration);
		 } else if (format == GST_FORMAT_BYTES) {
		 gst_query_set_duration(query, GST_FORMAT_BYTES,
						demuxer_info->segment.duration);
		 }
		 res = TRUE;
		 break;
	 }
 
	 case GST_QUERY_SEEKING:
	 {
		 GstFormat format;
		 format = GST_FORMAT_TIME;
		 gst_query_set_seeking(query,
						       format,
						       demuxer_info->seekable,
						       demuxer_info->segment.start,
						       demuxer_info->segment.stop);
 
		 res = TRUE;
		 break;
	 }
 
	 default:
	 res = FALSE;
	 }
	 gst_object_unref(demuxer_info);
	 return res;
 
 }

 /*=============================================================================
 FUNCTION:	 gst_mp4demuxer_get_src_query_types   
 
 DESCRIPTION:	 performs query on src pad.    
 
 ARGUMENTS PASSED:
		 pad	 -	 pointer to GstPad
		 query	 -	 pointer to GstQuery		
			 
 RETURN VALUE:
		 TRUE	 -	 success
		 FALSE	 -	 failure
 
 ==============================================================================*/
 
 static const GstQueryType *gst_mp4demuxer_get_src_query_types(GstPad * pad)
 {
	 static const GstQueryType types[] = {
	 GST_QUERY_DURATION,
	 GST_QUERY_SEEKING,
	 0
	 };
	 return types;
 }
 
 /*=============================================================================
 FUNCTION:		 mfw_gst_mp4demuxer_handle_src_event
 
 DESCRIPTION:	 Handles an event on the source pad.
 
 ARGUMENTS PASSED:
		 pad		-	 pointer to pad
		 event		-	 pointer to event
 RETURN VALUE:
		 TRUE		-	 if event is sent to source properly
		 FALSE		-	 if event is not sent to source properly
 =============================================================================*/
 
 static gboolean mfw_gst_mp4demuxer_handle_src_event(GstPad* src_pad, GstEvent* event)
 {
	 gboolean res = TRUE;
	 gint i;
 
	 MFW_GST_MP4DEMUX_INFO_T *demuxer_info = MFW_GST_MP4_DEMUXER(GST_PAD_PARENT (src_pad));
  
	 switch (GST_EVENT_TYPE (event)) 
	 {
		 case GST_EVENT_SEEK:
		 {
			 GstSeekFlags flags;
			 gboolean	  flush;
			 gdouble	  rate;
			 GstSeekType  cur_type;
			 GstSeekType  stop_type;
			 gint64 	  cur;
			 GstFormat	  format;
			 gint64 	  stop;
			
			 GST_DEBUG("Handling the seek event\n");
 
			 gst_event_ref (event);
 
			 /* parse a seek event */
			 gst_event_parse_seek(event, &rate, &format, &flags,
			 &cur_type, &cur, &stop_type, &stop);
			 
			 if (demuxer_info->do_seek_flag == TRUE && (demuxer_info->total_src_pad > 1)/*!strncmp("video", GST_PAD_NAME(src_pad), 5)*/)
			 {
				 
				 gst_event_unref (event);
				 return TRUE;
			 }


             demuxer_info->do_seek_flag = TRUE;
			 /* demuxer cannot do seek if format is not bytes */
			 if (format != GST_FORMAT_TIME)
			 {
				 GST_WARNING("SEEK event not TIME format.\n");
				 gst_event_unref (event);
				 return TRUE;
			 }
 
			 /* if the seek time is greater than the stream duration */
			 if(cur > demuxer_info->segment.duration) 
			 {
				 GST_WARNING("SEEK event exceed the maximum duration.\n");
				 cur = demuxer_info->segment.duration;
			 }	 
		 
			 flush = flags & GST_SEEK_FLAG_FLUSH;
 
			 if (flush)
			 {
				 guint strem_count;
 
				 /* sends an event to upstream elements with event as new flush start event*/
				 res = gst_pad_push_event (demuxer_info->sinkpad, gst_event_new_flush_start ());
				 if (!res) 
				 {
					 GST_ERROR("Failed to push event upstream!");
				 }
				 
				 for(strem_count = 0; strem_count < demuxer_info->total_src_pad; strem_count++)
				 {
					 if (demuxer_info->srcpad[strem_count]) {
						 /* sends an event to downstream elements with event as new flush start event*/ 				  
						 res = gst_pad_push_event (demuxer_info->srcpad[strem_count], 
												   gst_event_new_flush_start());
						 if (!res)
						 {
							 GST_ERROR("Failed to push event downstream!");
						 }
					 }
				 }
			 }/*if (flush) */ 
			 GST_WARNING("Stop all the task.\n");
			 /* stop all task for demuxer seeking, after seeking is performed
			  * start task again. */
			 for (i = 0; i < demuxer_info->total_src_pad; i++)
			 {
				 gst_pad_pause_task(demuxer_info->srcpad[i]);
			 }
			 /* sends an event to upstream elements with event as new flush stop event
				, to resume data flow */
			 res = gst_pad_push_event (demuxer_info->sinkpad, gst_event_new_flush_stop ());
			 
			 /*Lock the stream lock of sinkpad*/
			 GST_PAD_STREAM_LOCK(demuxer_info->sinkpad);
			 /* perform seek */
			 res = mfw_gst_mp4demuxer_seek(demuxer_info, event);
 
			 if(res == FALSE)
			 {
			   GST_ERROR("Failed in demuxer seek !!\n");
 
				/*Unlock the stream lock of sink pad*/	 
			   GST_PAD_STREAM_UNLOCK(demuxer_info->sinkpad);
			   return FALSE;
			 }
			 
			 if(res == FALSE)
			 {
				GST_ERROR("Failed to push event upstream!");
			 }
			 
			  /* send flush stop to down stream elements,to enable data flow*/
			 if(flush)
			 {
			   guint strem_count;
			   for(strem_count = 0; strem_count < demuxer_info->total_src_pad; strem_count++)
			   {
						   
				   res = gst_pad_push_event (demuxer_info->srcpad[strem_count],
										   gst_event_new_flush_stop());
				   if (!res) 
				   {
					   GST_ERROR("Failed to push event downstream!");
				   }
			   }
		   }
		  
			/*Unlock the stream lock of sink pad*/ 
			GST_PAD_STREAM_UNLOCK(demuxer_info->sinkpad);
			for (i = 0; i < demuxer_info->total_src_pad; i++)
			{
			 if (!strncmp("video", GST_PAD_NAME(demuxer_info->srcpad[i]), 5))
			 {
				 demuxer_info->seek_flag=TRUE;
				 demuxer_info->new_seg_flag_video=TRUE;
				 GST_DEBUG("seek mp4 video src_pad !\n");
				 res = gst_pad_start_task(demuxer_info->srcpad[i],
						  (GstTaskFunction)
						  mfw_gst_mp4demuxer_taskfunc,
						  demuxer_info->srcpad[i]);
				   if (!res) 
				   {
					   GST_ERROR("Failed to start task!\n");
				   }
			 }
			 if (!strncmp("audio", GST_PAD_NAME(demuxer_info->srcpad[i]), 5))
			 {
				 demuxer_info->new_seg_flag_audio=TRUE;
				 GST_DEBUG("seek mp4 audio src_pad !\n");	 
				 res = gst_pad_start_task(demuxer_info->srcpad[i],
						  (GstTaskFunction)
						  mfw_gst_mp4demuxer_taskfunc,
						  demuxer_info->srcpad[i]);

				   if (!res) 
				   {
					   GST_ERROR("Failed to start task!\n");
				   }
			 }
			}
 
	 break;
	 }
 
		 default:
		 res = FALSE;
		 break;
	 }
	 demuxer_info->do_seek_flag = FALSE; 
	 gst_event_unref (event);
	 return TRUE;
 }


/*=============================================================================
FUNCTION:			  mp4_write_audio_data
	 
DESCRIPTION:		  Pushes the parsed data to respective source pads.

ARGUMENTS PASSED:	  
source_buffer  -   pointer to the data which needs to be 
				  pushed to source pad.
total_size		  length/Size of the data pushed.
time_stamp	  -   The time stamp of each frame		
src_pad_index  -   the src pad no to which data has to be pushed
	 
	 

RETURN VALUE:		  
	   TRUE   -   if write_audio_data could be performed
	   FALSE  -   if write_audio_data could not be performed
	 
=============================================================================*/
static gboolean mp4_write_audio_data(MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer,
								  GstBuffer *gstbuf,
								  GstPad * pad, 
								  gint32 track_index)
{
    GstFlowReturn result = GST_FLOW_OK;
    GstClockTime stream_time = GST_BUFFER_TIMESTAMP(gstbuf); ;

    if (mp4_demuxer->new_seg_flag_audio == TRUE) 
    {
        GST_DEBUG("sending new segment event on to audio src pad \n ");
        gint64 new_start; 
		new_start = mp4_demuxer->segment.start;
        if (!gst_pad_push_event(pad,gst_event_new_new_segment(FALSE, 1.0, GST_FORMAT_TIME, new_start,
                GST_CLOCK_TIME_NONE, new_start))) 
        {
            GST_ERROR("\nCannot send new segment to the src pad\n");
            mp4_demuxer->new_seg_flag_audio = FALSE;
            goto Warrning_Msg;
        }
        mp4_demuxer->new_seg_flag_audio = FALSE;
    }

    /* get the timestamp of the data in the buffer, unit is ns */
    if(mp4_demuxer->tracks[track_index].audioCodec == AUDIO_PCM){
        GST_BUFFER_TIMESTAMP(gstbuf) = -1;
    }
 
    GST_DEBUG("pushing audio data on to src pad \n ");

    /* pushes buffer to the peer element */
    result = gst_pad_push(pad, gstbuf);
    if (result != GST_FLOW_OK) 
    {
        GST_ERROR("\n Cannot Push Data onto next element, %d\n", result);
        return FALSE;
    }

    /* Set the last observed stop position in the segment to position */
    gst_segment_set_last_stop(&mp4_demuxer->segment, GST_FORMAT_TIME, stream_time);
    return TRUE;

    Warrning_Msg:
    {
        GstMessage *message = NULL;
        GError *gerror = NULL;
        gerror = g_error_new_literal(1, 0, "pads are not negotiated!");

        message = gst_message_new_warning(GST_OBJECT(GST_ELEMENT(mp4_demuxer)), gerror,
                    "audio pads are not negotiated!");
        gst_element_post_message(GST_ELEMENT(mp4_demuxer), message);
        g_error_free(gerror);
        return TRUE;
    }
}


static void Add_NAL_StartCode(unsigned char *buf, gint32 data_size)
{
    unsigned char *temp_buf;
    guint32 jump, counter;

	counter = 0;
	temp_buf = buf;

    while (counter < data_size) 
    {        
        jump = (temp_buf[0]<<24)|(temp_buf[1]<<16)|(temp_buf[2]<<8)|(temp_buf[3]);
		//g_print("jump buffer offset is: %d\n", jump);
        if (jump==0x00000001) 
		{
            break;
        }
		else
		{
            temp_buf[2] = temp_buf[1] = temp_buf[0] = 0;
            temp_buf[3] = 0x01;
			temp_buf += (jump + 4);
			counter += (jump + 4);
		}
    }
}

 
/*=============================================================================
FUNCTION:			  mp4_write_video_data
DESCRIPTION:		  Pushes the parsed data to respective source pads.
ARGUMENTS PASSED:	  
    mp4_demuxer     - pointer to mp4 demuxer context
    gstbuf		    - pointer to gstbuffer needed to be pushed
    pad             - respective source pad
RETURN VALUE:		  
    TRUE   -   if write_video_data could be performed
    FALSE  -   if write_video_data could not be performed
================================================================================*/
static gboolean mp4_write_video_data(MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer,
								  GstBuffer *gstbuf,
								  GstPad * pad)
{
	GstCaps *src_caps = NULL;
    GstBuffer *specificBuf, *src_data_buf = NULL;
    gint total_size, src_buf_size = 0;	
    GstFlowReturn result = GST_FLOW_OK;
	unsigned char *tmp_buf = NULL;
    GstClockTime stream_time;

    stream_time = GST_BUFFER_TIMESTAMP(gstbuf); 
	total_size = GST_BUFFER_SIZE(gstbuf);
	specificBuf = mp4_demuxer->tracks[mp4_demuxer->video_track_decode].decoderSpecificInfo;

	if((specificBuf != NULL) && (mp4_demuxer->new_seg_flag_video == TRUE))
		src_buf_size = total_size + GST_BUFFER_SIZE(specificBuf);
	else
		src_buf_size = total_size;

    src_caps = GST_PAD_CAPS(pad);
    
    if (src_caps == NULL) {
        GST_ERROR("\n Caps not Correct \n");
        return FALSE;
    }
    
    /* Allocates a new, empty buffer optimized to push to pad stream_ptr->src_pad */
    result = gst_pad_alloc_buffer(pad, 0, src_buf_size, src_caps, &src_data_buf);

	if (result == GST_FLOW_WRONG_STATE) 
    {
        GST_ERROR("\n Cannot Create Output Buffer, result is %d", result);
        return FALSE;
    }
    else if (result == GST_FLOW_NOT_LINKED  ) 
    {
        GST_ERROR("\n Cannot Create Output Buffer, result is %d", result);
        return TRUE;
    }

	tmp_buf = GST_BUFFER_DATA(src_data_buf);

    if (mp4_demuxer->new_seg_flag_video == TRUE) 
    {
        /* after new segment comes, add codec data into input buffer  */
		if ((specificBuf != NULL) && (mp4_demuxer->tracks[mp4_demuxer->video_track_decode].videoCodec == VIDEO_H264))
		{
            /* copy data into src_data_buf */
			unsigned char *video_data_first = GST_BUFFER_DATA(gstbuf);
            unsigned char *video_data = GST_BUFFER_DATA(specificBuf);
            unsigned int video_data_size = GST_BUFFER_SIZE(specificBuf);
			/* change this frame's data to add start code */
			Add_NAL_StartCode(video_data_first, GST_BUFFER_SIZE(gstbuf));
            memcpy  (tmp_buf, video_data, video_data_size); 
            memcpy(tmp_buf + video_data_size, GST_BUFFER_DATA(gstbuf), total_size);
		}
		else
		{
			unsigned char *video_data = GST_BUFFER_DATA(gstbuf);
			unsigned int video_data_size = GST_BUFFER_SIZE(gstbuf);
			memcpy(tmp_buf, video_data, video_data_size);		
		}

	    {
			/* send new start to src pad */
            gint64 new_start;
            GST_DEBUG("sending new segment event on to video src pad \n ");	 
    		new_start = mp4_demuxer->segment.start;
    
            if (!gst_pad_push_event(pad,
                    gst_event_new_new_segment(FALSE, 1.0, GST_FORMAT_TIME, new_start,
                    GST_CLOCK_TIME_NONE, new_start))) 
            {
                GST_ERROR("\nCannot send new video segment to the src pad\n");
                mp4_demuxer->new_seg_flag_video = FALSE;
                goto War_Msg;
            }
	    }
        mp4_demuxer->new_seg_flag_video = FALSE;
    }
	else
	{
	    /* directly copy input data into pad buffer */
		/* copy data into src_data_buf */
		unsigned char *video_data = GST_BUFFER_DATA(gstbuf);
		unsigned int video_data_size = GST_BUFFER_SIZE(gstbuf);
		memcpy(tmp_buf, video_data, video_data_size); 		
	}
	
    /* set timestamp and sync, then buffer unref for input buffer */ 
	GST_BUFFER_TIMESTAMP(src_data_buf) = stream_time;

	if(GST_BUFFER_FLAG_IS_SET(gstbuf,GST_BUFFER_FLAG_IS_SYNC))
		GST_BUFFER_FLAG_SET(src_data_buf, GST_BUFFER_FLAG_IS_SYNC);
    
    gst_buffer_unref(gstbuf);
    /* get the timestamp of the data in the buffer */
    GST_DEBUG("pushing video data on to src pad: %d\n", GST_BUFFER_SIZE(src_data_buf));

    /* pushes buffer to the peer element */
    result = gst_pad_push(pad, src_data_buf);
    if (result != GST_FLOW_OK) {
        GST_ERROR("\n Cannot Push video Data onto next element, reason is %d", result);
        return FALSE;
    }
    mp4_demuxer->videosent = TRUE;
    /* Set the last observed stop position in the segment to position */
    gst_segment_set_last_stop(&(mp4_demuxer->segment), GST_FORMAT_TIME, stream_time);
    return TRUE;

War_Msg:
    {
        GstMessage *message = NULL;
        GError *gerror = NULL;
        gerror = g_error_new_literal(1, 0, "pads are not negotiated!");

        message = gst_message_new_warning(GST_OBJECT(GST_ELEMENT(mp4_demuxer)), gerror,
                    "video pads are not negotiated!");
        gst_element_post_message(GST_ELEMENT(mp4_demuxer), message);
        g_error_free(gerror);
        return TRUE;
    }
}
/*=============================================================================
FUNCTION:            mp4_demuxer_create_audio_caps
        
DESCRIPTION:         setup audio caps for the audio src pad.

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
        
  
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd
        
=============================================================================*/
static gboolean mp4_demuxer_create_audio_caps(MFW_GST_MP4DEMUX_INFO_T *demuxer, gint src_pad_idx)
{

	/* mp4 is microsoft format, so both audio and video should label it */
	gint audio_track, version, endiness;
    const char *media_type;
    GstPadTemplate *templ = NULL;
    gchar *padname = NULL;	
    gboolean signed_pcm, type_find = TRUE; 
    guint sampling_frequency = 0;
	GstCaps *caps = NULL;
	endiness = 0;
	signed_pcm = TRUE;
	audio_track = demuxer->audio_track_decode;
	version = 0;
	demuxer->srcpadnum2trackindex[src_pad_idx].trackindex = -1;
	if(audio_track == -1)
		return FALSE;
	if ((demuxer->tracks[audio_track].audioCodec== AUDIO_MP3) 
		|| (demuxer->tracks[audio_track].audioCodec == AUDIO_AAC))
	{
		GST_DEBUG("\n Audio Media type is MPEG");
		media_type="audio/mpeg";  
		version = 1;
		if((demuxer->tracks[audio_track].audioCodec == AUDIO_AAC))
			version = 4;
		else if((demuxer->tracks[audio_track].audioCodec == AUDIO_MPEG2_AAC))
			version = 2;
		sampling_frequency = demuxer->tracks[audio_track].sampleRate;
	}
	else if (demuxer->tracks[audio_track].audioCodec == AUDIO_AMR)
	{
		GST_DEBUG("\n Audio Media type is AMR");
		if(demuxer->tracks[audio_track].decoderSubtype == AUDIO_AMR_NB)
		    media_type="audio/AMR";
		else
			media_type="audio/AMR-WB";
		sampling_frequency = demuxer->tracks[audio_track].sampleRate;;
	}
	else if (demuxer->tracks[audio_track].audioCodec == AUDIO_PCM)
	{
		GST_DEBUG("\n Audio Media type is PCM");
		media_type="audio/x-raw-int";
		if(demuxer->tracks[audio_track].decoderSubtype == AUDIO_PCM_U8)
		{
			endiness = G_BYTE_ORDER;
            signed_pcm = FALSE;
		}
		else if(demuxer->tracks[audio_track].decoderSubtype == AUDIO_PCM_S16LE)
		{
			endiness = G_LITTLE_ENDIAN;
            signed_pcm = TRUE;
		}
		else if(demuxer->tracks[audio_track].decoderSubtype == AUDIO_PCM_S16BE)
		{
			endiness = G_BIG_ENDIAN;
            signed_pcm = TRUE;
		}		
		sampling_frequency = demuxer->tracks[audio_track].sampleRate;;
	}
	else 
	{
		GST_ERROR("\n audio type is unknown %d\n", 
			demuxer->tracks[audio_track].audioCodec);
		type_find = FALSE;
	}
	/* defines the capability for audio pad */
	if (type_find) 
	{
    	/* Defines the audio src pad template */
    		//g_print("audio type find\n");
    		templ = audio_src_templ();
    	if ((demuxer->tracks[audio_track].audioCodec== AUDIO_MP3) 
    		|| (demuxer->tracks[audio_track].audioCodec == AUDIO_AAC))
    	{
    		caps = gst_caps_new_simple(media_type,
    				   "mpegversion", G_TYPE_INT, version,
    				   "bitrate", G_TYPE_INT,
    				   demuxer->tracks[audio_track].bitRate,
    				   "rate", G_TYPE_INT,sampling_frequency,
    				   "channels",G_TYPE_INT,demuxer->tracks[audio_track].numChannels,
    				   NULL);	
    	}
    	else if(demuxer->tracks[audio_track].audioCodec == AUDIO_PCM)
    	{
		    caps = gst_caps_new_simple(media_type, 
				   "rate", G_TYPE_INT,sampling_frequency,     /* sample rate */
				   "channels",G_TYPE_INT, demuxer->tracks[audio_track].numChannels,
				   "width", G_TYPE_INT, demuxer->tracks[audio_track].pcm_width,
				   "depth", G_TYPE_INT, demuxer->tracks[audio_track].pcm_width,
                   "endianness", G_TYPE_INT, endiness,
                   "signed", G_TYPE_BOOLEAN, signed_pcm,				   
				   NULL);

			//g_print("%s\n", gst_caps_to_string(caps));
    	}
		else
    		caps = gst_caps_new_simple(media_type,
    				   "bitrate", G_TYPE_INT,
    				   demuxer->tracks[audio_track].bitRate,
    				   "rate", G_TYPE_INT,sampling_frequency,
    				   "channels",G_TYPE_INT,demuxer->tracks[audio_track].numChannels,
    				   NULL);
		if (!caps) 
		{
			 GST_ERROR("audio type find, but can not create caps\n");
			 return FALSE;
		}	

		//g_print("audio set cap %s %d\n", gst_caps_to_string(caps), src_pad_idx);
    	demuxer->caps[src_pad_idx] = caps;
		/* Creates a new pad with the name padname from the template templ */		
		padname = g_strdup_printf(PARSER_AUDIOPAD_TEMPLATE_NAME, audio_track);
	    demuxer->srcpad[src_pad_idx] = gst_pad_new_from_template(templ, padname);
		if(padname != NULL)
		{
			g_free(padname);
			padname = NULL;
		}		
	}
	else
		return FALSE;
	
	demuxer->srcpadnum2trackindex[src_pad_idx].trackType = MEDIA_AUDIO;
	demuxer->srcpadnum2trackindex[src_pad_idx].trackindex = audio_track;
	return TRUE;
}

/*=============================================================================
FUNCTION:            mp4_demuxer_create_video_caps
        
DESCRIPTION:         setup video caps for the video src pad.

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
        
  
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd
        
=============================================================================*/
static gboolean mp4_demuxer_create_video_caps(MFW_GST_MP4DEMUX_INFO_T *demuxer, gint src_pad_idx)
{

	/* mp4 is microsoft format, so both audio and video should label it */
	gint video_track, version;
    GstPadTemplate *templ = NULL;	
	gchar *padname = NULL;
    const char *media_type;
    guint frame_width;
    guint frame_height;
    double frame_rate;	
    gboolean type_find = TRUE;
	GstCaps *caps = NULL;	
	
	version = 0;	
	video_track = demuxer->video_track_decode;
	demuxer->srcpadnum2trackindex[src_pad_idx].trackindex = -1;
	if(video_track == -1)
		return FALSE;
	frame_width = demuxer->tracks[video_track].width;
	frame_height = demuxer->tracks[video_track].height;
	frame_rate = demuxer->tracks[video_track].video_framerate;
	
    if(demuxer->tracks[video_track].videoCodec ==VIDEO_MPEG4)
	{
		GST_DEBUG("\n Video Media type is mpeg4");
		media_type = "video/mpeg";
	}
	else if (demuxer->tracks[video_track].videoCodec == VIDEO_H264)
	{
		GST_DEBUG("\n Video Media type is H.264");
		media_type = "video/x-h264";
	}
	else if(demuxer->tracks[video_track].videoCodec ==VIDEO_H263)
	{
		GST_DEBUG("\n Video Media type is xvid video");
		media_type = "video/x-h263";
	}

	else if (demuxer->tracks[video_track].videoCodec == VIDEO_MJPG
		     || demuxer->tracks[video_track].videoCodec == VIDEO_JPEG)
	{
		GST_DEBUG("\n Video Media type is motion jpeg file");
		media_type = "image/jpeg";
	}
	else 
	{
		type_find = FALSE;
		GST_ERROR("\n mp4 video type is unknown, %d\n", demuxer->tracks[video_track].videoCodec);
	}
	
	if ( ((frame_width > MAX_RESOLUTION_WIDTH) || (frame_width < MIN_RESOLUTION_WIDTH)) 
	  && ((frame_height > MAX_RESOLUTION_HEIGHT)||(frame_height < MIN_RESOLUTION_HEIGHT)))
	{
		GST_ERROR("Unsupported video resolution!\n");
		return FALSE;
	}
		
	if (type_find)
	{
		/* Defines the video src pad template */
		templ = video_src_templ();
	   // g_print("video type find\n");
		/* defines the capability for video pad */
	   if(demuxer->tracks[video_track].videoCodec ==VIDEO_MPEG4)
	   	{
	        caps = gst_caps_new_simple(media_type,
				       "systemstream", G_TYPE_INT, 0,
					   "mpegversion", G_TYPE_INT, 4,
					   "height", G_TYPE_INT, frame_height,
					   "width", G_TYPE_INT, frame_width,
					   "framerate", GST_TYPE_FRACTION, (gint)(frame_rate*1000), 1000,
					   NULL); 
	   	}
	   else
		   caps = gst_caps_new_simple(media_type,
		              "systemstream", G_TYPE_INT, 0,
					  "height", G_TYPE_INT, frame_height,
					  "width", G_TYPE_INT, frame_width,
					  "framerate", GST_TYPE_FRACTION, (gint)(frame_rate*1000), 1000,
					  NULL); 
	   	
	   if(demuxer->tracks[video_track].decoderSpecificInfo != NULL)
	     if(GST_BUFFER_SIZE(demuxer->tracks[video_track].decoderSpecificInfo) > 0)
	        gst_caps_set_simple(caps, "codec_data", GST_TYPE_BUFFER, demuxer->tracks[video_track].decoderSpecificInfo, NULL);
		if (!caps)
		{
			 GST_ERROR("void type find, but can not create caps: %d\n", demuxer->tracks[video_track].videoCodec);
			 return FALSE;
		}
		demuxer->caps[src_pad_idx] = caps;
		padname = g_strdup_printf(PARSER_VIDEOPAD_TEMPLATE_NAME, video_track);
	    demuxer->srcpad[src_pad_idx] = gst_pad_new_from_template(templ, padname);
		if(padname != NULL)
		{
			g_free(padname);
			padname = NULL;
		}	
	}
	else
		return FALSE;
	demuxer->srcpadnum2trackindex[src_pad_idx].trackType = MEDIA_VIDEO;
	demuxer->srcpadnum2trackindex[src_pad_idx].trackindex = video_track;
	
	return TRUE;
}

/*=============================================================================
FUNCTION:            mp4_demuxer_create_pad
        
DESCRIPTION:         create pad and connected with the caps.

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
        
  
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd
        
=============================================================================*/
static gboolean mp4_demuxer_create_pad(MFW_GST_MP4DEMUX_INFO_T *demuxer, gint src_pad_idx)
{
    gboolean set;
	GstPad *pad;
	GstCaps *caps = NULL;	
	//g_print("enter mp4_demuxer_create_pad\n");


	set = TRUE;	
    caps = demuxer->caps[src_pad_idx];
	pad = demuxer->srcpad[src_pad_idx];	
	
	gst_pad_use_fixed_caps(pad);
	
	/* set the stream_ptr on to the pad */
	gst_pad_set_element_private(pad, demuxer->srcpad[src_pad_idx]);
	
	/* set the capabilities of the pad */
	set = gst_pad_set_caps(pad, caps);
	if (set == FALSE) {
		GST_ERROR
		("\n unable to set the capability of the pad:result is %d\n",
		 set);
		return set;
	}
	
	//g_print("enter mp4_demuxer_create_pad1\n");
	/* check if the caps represents media formats */
	if (gst_caps_is_empty(gst_pad_get_caps(pad)))
	{
		GST_ERROR("\n caps is empty\n");
			gst_caps_unref(caps);
			return FALSE;
	}
	/* Activates the given pad */
	if (gst_pad_set_active(pad, TRUE) != TRUE) {
		GST_ERROR("\nThe pad was activated failed\n");
		return FALSE;
	}
	
	//g_print("enter mp4_demuxer_create_pad2\n");	
	/* adds a pad to the element */
	if(gst_element_add_pad(GST_ELEMENT(demuxer), pad) != TRUE)
	{
		GST_ERROR("failed add pad to element\n");
		return FALSE;
	}
	
	GST_LOG_OBJECT(GST_ELEMENT(demuxer), "Added pad %s with caps %p",
				GST_PAD_NAME(pad), caps);
	
	/* unref a GstCap, because gst_pad_set_caps ref caps, so should unref it
	 * as soon as possible, otherwise, it will not be freed. */
	gst_caps_unref(caps);

	return set;
}

/*=============================================================================
FUNCTION:            mp4_demuxer_setup_caps
        
DESCRIPTION:         Add and Creates caps for src pad.

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
                     total_tracks - Total number of tracks present in the file. 
        
  
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd
        
=============================================================================*/


static gboolean mp4_demuxer_setup_caps(MFW_GST_MP4DEMUX_INFO_T *demuxer)
{

    guint src_pad_idx = 0;
    gboolean ret = TRUE; 
	
	g_print("total src pad is:%d\n", demuxer->total_src_pad);
    if(demuxer->audio_track_decode != -1)
    {
        ret = mp4_demuxer_create_audio_caps(demuxer, src_pad_idx);
    	if(ret != TRUE)
    	{
    		GST_ERROR("audio type find, but can not create caps\n");
    		goto Err_Msg;
    	}
    	//g_print("success for mp4_demuxer_create_audio_caps\n");	
    	
    
        ret = mp4_demuxer_create_pad(demuxer, src_pad_idx);
    
    	if(ret != TRUE)
    	{
    		GST_ERROR("can not create pad for audio caps\n");
    		goto Err_Msg;
    	}
		src_pad_idx++;
    }		
    if(demuxer->video_track_decode != -1)
    {
        ret = mp4_demuxer_create_video_caps(demuxer, src_pad_idx);
    	if(ret != TRUE)
    	{
    		GST_ERROR("video type find, but can not create caps\n");
    		goto Err_Msg;
    	}    	
        ret = mp4_demuxer_create_pad(demuxer, src_pad_idx);
    
    	if(ret != TRUE)
    	{
    		GST_ERROR("can not create pad for video caps\n");
    		goto Err_Msg;
    	}		
    	src_pad_idx++;
    }
        
    return TRUE;

    /* send an error message for an unrecoverable error */
Err_Msg:
    {
    GstMessage *message = NULL;
    GError *gerror = NULL;

    gerror = g_error_new_literal(1, 0, "mp4_demuxer_setup_caps error");

    message =
        gst_message_new_error(GST_OBJECT(GST_ELEMENT(demuxer)), gerror,
                  "debug none");
    gst_element_post_message(GST_ELEMENT(demuxer), message);
    g_error_free(gerror);
    return FALSE;
    }

}
/*=============================================================================
FUNCTION:            mp4_demuxer_set_pad_caps
        
DESCRIPTION:         Reset the capabilities of the pads created before.

ARGUMENTS PASSED:    demuxer      - The main structure of mp4 parser plugin context.
                     total_tracks - Total number of tracks present in the file. 
        
  
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd
        
=============================================================================*/

static gboolean mp4_demuxer_set_pad_caps(MFW_GST_MP4DEMUX_INFO_T * demuxer,
                     guint32 total_tracks)
{
    guint pad_count;
    gboolean set;

    for (pad_count=0; pad_count<demuxer->total_src_pad; pad_count++)
	{
        if ((demuxer->srcpad[pad_count]) && (demuxer->caps[pad_count]))
		{
            GST_DEBUG("set track:%d.\n",pad_count);
            set = gst_pad_set_caps(demuxer->srcpad[pad_count],
                                    demuxer->caps[pad_count]);
            if (set == FALSE) 
			{
                GST_ERROR("\n unable to set the capability of the pad:result is %d\n", set);
                return FALSE;
            }
        }
    }
    
    return TRUE;
}


/*=============================================================================
FUNCTION:            mfw_gst_mp4demuxer_sink_event
        
DESCRIPTION:         This functions handles the events that triggers the
                     sink pad of the mp4demuxer element.
  

ARGUMENTS PASSED:
        pad      -   pointer to pad
        event    -   pointer to event
        
  
RETURN VALUE:
       TRUE       -  if event is sent to sink properly
       FALSE      -  if event is not sent to sink properly

=============================================================================*/

static gboolean mfw_gst_mp4demuxer_sink_event(GstPad * pad,
                          GstEvent * event)
{

    gboolean result = TRUE;

    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_NEWSEGMENT:
    {
        GstFormat format;
        gst_event_parse_new_segment(event, NULL, NULL, &format, NULL,
                    NULL, NULL);
        if (format == GST_FORMAT_TIME) {
        GST_LOG("\nCame to the FORMAT_TIME call\n");
        gst_event_unref(event);
        result = TRUE;
        } else {
        GST_LOG("dropping newsegment event in format %s",
            gst_format_get_name(format));
        gst_event_unref(event);
        result = TRUE;
        }
        break;
    }

    case GST_EVENT_EOS:
    {
        GST_DEBUG("\nDemuxer: Sending EOS to Decoder\n");
        result = gst_pad_push_event(pad, event);

        if (result != TRUE) {
        GST_ERROR("\n Error in pushing the event, result is %d\n",
              result);
        }
        break;
    }

    default:
    {
        result = gst_pad_event_default(pad, event);
        return TRUE;
    }

    }

    return result;
}


/*=============================================================================
FUNCTION:            mfw_gst_mp4demuxer_parse
        
DESCRIPTION:         This is the main functions, where calls to MP4 Parser 
                     are made.

ARGUMENTS PASSED:    demuxer - The context of the main mp4 demuxer element.
        

        
=============================================================================*/

static gboolean mfw_gst_mp4demuxer_parse(MFW_GST_MP4DEMUX_INFO_T * demuxer)
{

    GstPad *src_pad;
    gboolean ret_value;
    guint pad_count;      /* track counter */
    guint total_tracks;     /* total number of tracks */
    guint64 duration;     /* stream duration */ 


	MP4_PARSER_ERROR_CODE err = 0; 
	
	//g_print("\n, enter parser module\n");
	err = mfw_gst_mp4demuxer_parser_init(demuxer);
	if (err != PARSER_SUCCESS) 
	{
		GST_ERROR("\nFATAL ERROR during mp4 parser init\n");
		goto Err_Msg;
	}

    err = MP4GetNumTracks(demuxer->parserHandle, &(demuxer->total_tracks));
	if (err != PARSER_SUCCESS) {
		GST_DEBUG("\n Error in get num tracks %d\n", err);
		goto Err_Msg;
	}	

	err = mp4_demuxer_fill_stream_info(demuxer);
	if(PARSER_SUCCESS != err)
	{
		GST_DEBUG("fail to get mp4 file stream info\n");
		goto Err_Msg;
	}

    if (demuxer->stop_request == FALSE) 
	{
        /**** Setting Pads for the streams present ****/
        ret_value = mp4_demuxer_setup_caps(demuxer);
        if (PARSER_SUCCESS != err) 
		{
            GST_ERROR("Error in setting the pad capability in fill stream info\n");
            goto Err_Msg;
        }
    } 
	else 
	{
        /* reseting the capabilities of the pad */
        ret_value = mp4_demuxer_set_pad_caps(demuxer, total_tracks);
        if (!ret_value) {
            GST_ERROR("Error in setting the pad capability \n");    
            goto Err_Msg;
            
        }
    }

    /* Signalling no more pads to the application, all the pad should be created
     * at this point */
    GST_DEBUG("signaling no more pads from mfw_mp4demuxer");
    gst_element_no_more_pads (GST_ELEMENT (demuxer));

    /* duration of the mp4 file */
    duration=demuxer->usDuration;

    /* Set the duration of the segment to duration in nanoseconds */
    gst_segment_set_duration(&demuxer->segment,
                     GST_FORMAT_TIME,
                     duration * 1000);
   
    /* register the query functions on to the src pad created */
    for (pad_count = 0; pad_count < demuxer->total_src_pad; pad_count++) 
	{
    	gst_pad_set_query_type_function(demuxer->srcpad[pad_count],
    					GST_DEBUG_FUNCPTR
    					(gst_mp4demuxer_get_src_query_types));
    	gst_pad_set_query_function(demuxer->srcpad[pad_count],
    				   GST_DEBUG_FUNCPTR
    				   (mfw_gst_mp4demuxer_handle_src_query));
    
    	gst_pad_set_event_function (demuxer->srcpad[pad_count],GST_DEBUG_FUNCPTR (
		                          mfw_gst_mp4demuxer_handle_src_event));

        demuxer->padcache[pad_count].cachesize = 
            ((demuxer->tracks[demuxer->srcpadnum2trackindex[pad_count].trackindex].maxSampleSize>DEFAULT_MAX_MP4_SAMPLESIZE)?
                demuxer->tracks[demuxer->srcpadnum2trackindex[pad_count].trackindex].maxSampleSize:DEFAULT_MAX_MP4_SAMPLESIZE);
		if(demuxer->padcache[pad_count].cachesize > LIMIT_MAX_MP4_SAMPLESIZE)
		{
            GST_DEBUG("big sample size for mp4 file found: %d\n", demuxer->padcache[pad_count].cachesize);
			demuxer->padcache[pad_count].cachesize = LIMIT_MAX_MP4_SAMPLESIZE;
		}
        //g_print("max size %d = %d\n", pad_count, demuxer->padcache[pad_count].cachesize);
    }

    if(demuxer->video_track_decode != -1)
    {
        /* the video new segment flag should be set to TRUE,as new segment should 
           not be sent along with video object information */ 
         demuxer->new_seg_flag_video=TRUE;
    }
    if(demuxer->audio_track_decode != -1)
    {
    
     /* the audio new segment flag should be set to TRUE,as new segment should not be
        sent along with amr header information */ 
        demuxer->new_seg_flag_audio=TRUE;
    }

    /* Add the user data information to the taglist */
    MP4CreateTagList(demuxer);
     /* seek to the beginning of the movie */
	 {
        guint64  usSeekTime = 0;
		if(demuxer->audio_track_decode != -1)
		{
           err = MP4Seek(demuxer->parserHandle, demuxer->audio_track_decode, &usSeekTime, SEEK_FLAG_NO_LATER);
           if(PARSER_SUCCESS != err) 
           {
                   goto Err_Msg;
           }
   		   g_print("successfully seek audio the track %d\n", demuxer->audio_track_decode);
		}
		if(demuxer->video_track_decode != -1)
		{
           err = MP4Seek(demuxer->parserHandle, demuxer->video_track_decode, &usSeekTime, SEEK_FLAG_NO_LATER);
           if(PARSER_SUCCESS != err) 
           {
                   goto Err_Msg;
           }
   		   g_print("successfully seek video the track %d\n", demuxer->video_track_decode);
		}
		
	 }

    return TRUE;

    /* send an error message for an unrecoverable error */
Err_Msg:
    {
    GstMessage *message = NULL;
    GError *gerror = NULL;
    gerror = g_error_new_literal(1, 0, "mfw_gst_mp4demuxer_parse error");

    message =
        gst_message_new_error(GST_OBJECT(GST_ELEMENT(demuxer)), gerror,
                  "debug none");
    gst_element_post_message(GST_ELEMENT(demuxer), message);
    g_error_free(gerror);
    return FALSE;
    }
 }


#define STORE_CACHE_BUFFER(cache, buffer, value)\
    (cache).gstbuf = (buffer);\
    (cache).offset = (value);

#define GET_CACHE_BUFFER(cache, buffer)\
    buffer = (cache).gstbuf;\
    (cache).gstbuf = NULL;

#define TIMESTAMP_MP42GST(mp4time)\
    ((mp4time)*1000)

static GstBuffer * mp4_demuxer_get_one_sample(MFW_GST_MP4DEMUX_INFO_T * demux, gint idx, gint track)
{
	GstBuffer * parent=NULL, * son=NULL;
    GstPad * pad = demux->srcpad[idx];
    void * data;
    gint cachesize, offset;
	GstFlowReturn ret;
    MP4_PARSER_ERROR_CODE   Err;
	guint64 usStartTime;
	guint64 usEndTime;
	guint32 get_flag = 0;

    GET_CACHE_BUFFER(demux->padcache[idx], parent);
    cachesize = demux->padcache[idx].cachesize;
    offset = demux->padcache[idx].offset;

    g_mutex_lock(demux->media_file_lock);


    do {
        void * data;
        gint space, samplesize;
		GstBuffer *son1;

        if (parent==NULL){
            ret = gst_pad_alloc_buffer(pad, 0, cachesize, NULL, &parent);     
            if (ret != GST_FLOW_OK){
                GST_ERROR("alloc buffer failed, pause task! %d\n", ret);
                goto error;
            }
            offset = 0;
        }

        data = GST_BUFFER_DATA(parent)+offset;
        samplesize = space = GST_BUFFER_SIZE(parent)-offset;

        Err = MP4GetNextSample(demux->parserHandle, 
                                track,
                                data,  
                                &samplesize, 
                                &usStartTime, 
                                &usEndTime,
                                &get_flag);

        if ((Err == PARSER_EOS) || ((PARSER_SUCCESS != Err) && (PARSER_ERR_CONCEAL_FAIL != Err))){   
            ret = mfw_gst_mp4demuxer_sink_event(pad, gst_event_new_eos());
            if (ret != TRUE) {
                GST_ERROR("Error in pushing the event, result is %d\n", ret);
            }
            STORE_CACHE_BUFFER(demux->padcache[idx], NULL, 0);
            demux->eos_flag[idx]=TRUE;     
            goto error;

        }else{
			son1 = gst_buffer_create_sub(parent, offset, samplesize);
			if (son1==NULL){
				GST_ERROR("error on create sub gstbuffer\n");
				goto error;
			}
            if (space-samplesize>DEFAULT_MIN_MP4_SAMPLESIZE){
                offset += samplesize;
            }else{
                gst_buffer_unref(parent);
                parent = NULL;           
            }
			if (son){
                    son = gst_buffer_join(son, son1);
            }else
            	son=son1;
        }
    }while(get_flag & FLAG_SAMPLE_NOT_FINISHED);

    g_mutex_unlock(demux->media_file_lock);
    
    if (son){
        if (get_flag & FLAG_SYNC_SAMPLE){
            GST_BUFFER_FLAG_SET(son, GST_BUFFER_FLAG_IS_SYNC);
        }
        GST_BUFFER_TIMESTAMP(son) = TIMESTAMP_MP42GST(usStartTime);
            
    }

    STORE_CACHE_BUFFER(demux->padcache[idx], parent, offset);
    
    return son; 
    
error:
    if (son){
        gst_buffer_unref(son);
    }
    if (parent){
        gst_buffer_unref(parent);
    }
    g_mutex_unlock(demux->media_file_lock);
    return NULL; 
}

/*=============================================================================
FUNCTION:            mfw_gst_mp4demuxer_taskfunc
        
DESCRIPTION:         
                     The taskfunction reads tracks of video/audio data from 
                     the container,pushes it to the respective pad, until reach
                     file end.

ARGUMENTS PASSED:    srcpad - The track's pad pointer.
        

=============================================================================*/
static void mfw_gst_mp4demuxer_taskfunc(GstPad    *src_pad)
{
    
    gboolean  ret_value;
    guint     track_index;     /* indicate current pad's decoding track */
    guint     pad_count;
    
    MFW_GST_MP4DEMUX_INFO_T *demuxer;
	GstBuffer * gstbuf;

    demuxer =  MFW_GST_MP4_DEMUXER(GST_PAD_PARENT(src_pad));

    if(demuxer->seek_flag==TRUE)
        demuxer->seek_flag=FALSE;

    for (pad_count = 0; pad_count < MAX_SRC_PADS; pad_count++)
    {
        if (demuxer->srcpad[pad_count] == src_pad)
        {
            track_index = demuxer->srcpadnum2trackindex[pad_count].trackindex;
            break;
        }
    }
	
    gstbuf = mp4_demuxer_get_one_sample(demuxer, pad_count, track_index);

    if (gstbuf==NULL){
        
        GST_ERROR("Can not create one sample\n");
        goto pause_task;

    }

    //gst_buffer_set_caps(gstbuf, GST_PAD_CAPS(src_pad));
    
    if (demuxer->tracks[track_index].mediaType == MEDIA_VIDEO) 
    {
        /* push the parsed audio data to video src pad  */
        ret_value = mp4_write_video_data(demuxer,
                                         gstbuf,
                                         src_pad);
        if (ret_value == FALSE)
		{
            GST_ERROR("\n error in writing the video data:result is %d\n", ret_value);
            goto pause_task;
        }
    }
    else if (demuxer->tracks[track_index].mediaType == MEDIA_AUDIO) 
    {
		gst_buffer_set_caps(gstbuf, GST_PAD_CAPS(src_pad));    
        /* push the parsed audio data to audio src pad  */
        ret_value = mp4_write_audio_data(demuxer,
                                         gstbuf,
                                         src_pad,
                                         track_index);
        if (ret_value == FALSE) 
		{
            GST_ERROR("\n error in writing the audio data:result is %d\n", ret_value);
            goto pause_task;
        }
    }
    if (demuxer->videosent) 
	{
        /* Keep the balance of demuxer and codec */    
        usleep(MP4_DEMUX_WAIT_INTERVAL);
        demuxer->videosent = FALSE;
    }
    return;
    /*when got eof or error*/
pause_task:
    /* Pause the task actived on the pad */
    ret_value = gst_pad_pause_task(src_pad);
    if (FALSE == ret_value)
        GST_ERROR("\n There is no task on this src pad !! \n");
    
    return;
    
}
/*=============================================================================
FUNCTION:            mp4_demuxer_get_usr_data
        
DESCRIPTION:         

ARGUMENTS PASSED:    get usr data from file, author, name, etc
        
  
RETURN VALUE:
       TRUE       -  if the function is performed properly
       FALSE      -  if there is any errors encounterd
        

=============================================================================*/
static MP4_PARSER_ERROR_CODE mp4_demuxer_get_usr_data(MFW_GST_MP4DEMUX_INFO_T *demuxer)
{
   guint8 *userData;
   guint32 userDataSize, cnt;
   MP4_PARSER_ERROR_CODE err;
   UserDataID id;
    err = PARSER_SUCCESS;
	for(cnt = USER_DATA_TITLE; cnt <= USER_DATA_RATING; cnt++)
	{
	    id = (UserDataID)cnt;
		userData = NULL;
		userDataSize = 0;
    	err = MP4GetUserData(demuxer->parserHandle, id, &userData, &userDataSize);
        if(userData && (userDataSize > 0))
        {
            demuxer->usrdata[cnt].data_ptr = app_MP4LocalMalloc(userDataSize);
    		if(demuxer->usrdata[cnt].data_ptr == NULL)
    			return PARSER_INSUFFICIENT_MEMORY;
    		memcpy(demuxer->usrdata[cnt].data_ptr, userData, userDataSize);
    		demuxer->usrdata[cnt].data_size = userDataSize;
    		demuxer->usrdata[cnt].id = id;
        }
    	else
    	{
    		demuxer->usrdata[cnt].data_ptr = NULL;
    		demuxer->usrdata[cnt].data_size = 0;
    		demuxer->usrdata[cnt].id = 0;
    	}
	}

	return PARSER_SUCCESS;

}

/*=============================================================================
FUNCTION:            mfw_gst_mp4demuxer_activate

DESCRIPTION:         it will call gst_pad_activate_pull which activates or
                     deactivates the given pad in pull mode via dispatching to the
                     pad's activepullfunc

ARGUMENTS PASSED:
         pad       - pointer to GstPad to activate or deactivate

RETURN VALUE:
         TRUE      - if operation was successful
         FALSE     - if operation was unsuccessful

       
=============================================================================*/

static gboolean mfw_gst_mp4demuxer_activate(GstPad * pad)
{
    if (gst_pad_check_pull_range(pad)) 
	{
        return gst_pad_activate_pull(pad, TRUE);
    } 
	else 
	{
        return FALSE;
    }

    return TRUE;

}

/*=============================================================================
FUNCTION:            mfw_gst_mp4demuxer_activate_pull

DESCRIPTION:         This will start the demuxer task function in "pull" based  
                     scheduling model.

ARGUMENTS PASSED:
    pad              sink pad, where the data will be pulled into, from 
                     file source.
    active           pad is active or not(TRUE/FALSE)
 

RETURN VALUE:       
        TRUE      -  if operation was successful
        FALSE     -  if operation was unsuccessful

=============================================================================*/

static gboolean
mfw_gst_mp4demuxer_activate_pull(GstPad * pad, gboolean active)
{
    gboolean ret_val = TRUE;
    MFW_GST_MP4DEMUX_INFO_T *demuxer_data;
    gint pad_count;
	
    demuxer_data = MFW_GST_MP4_DEMUXER(GST_PAD_PARENT(pad));    
    if (active) 
    {
        
        /* read the track information present in the container */
        ret_val=mfw_gst_mp4demuxer_parse(demuxer_data);
        
        if(ret_val==FALSE)
        {
        /* if the parsing failed due to unsupported stream or 
            corrupted stream,return back */
            mfw_gst_mp4_demuxer_close(demuxer_data);
            return FALSE;
        }
        /* start a Task, which will read each frame of audio/video data from the 
        container */

        for (pad_count = 0; pad_count < demuxer_data->total_src_pad; pad_count++) 
		{
            if (demuxer_data->srcpad[pad_count])
			{
                ret_val = gst_pad_start_task(demuxer_data->srcpad[pad_count],
                        (GstTaskFunction)mfw_gst_mp4demuxer_taskfunc,
                        demuxer_data->srcpad[pad_count]);
            }
			if (ret_val == FALSE) 
			{
				GST_ERROR("Task could not be started \n");
				mfw_gst_mp4_demuxer_close(demuxer_data);
				return ret_val;
			}
        }
        return TRUE;
    }/* if (active)*/
    else 
    {

        for (pad_count = 0; pad_count < demuxer_data->total_src_pad; pad_count++)
        {
            if (demuxer_data->srcpad[pad_count]) 
			{
                ret_val = gst_pad_pause_task(demuxer_data->srcpad[pad_count]);
                if (ret_val == FALSE) 
				{
                    GST_ERROR("Task could not be paused \n");
                    return ret_val;
                }
            }
        }
        return TRUE;
    }
}

/*=============================================================================
FUNCTION:            MP4CreateTagList
    
DESCRIPTION:         This function Add the user data information of the mp4 container
                     to the taglist.

ARGUMENTS PASSED:
      demuxer_info   : - The main structure of mp4 parser plugin context.
      
     
        
=============================================================================*/

static void MP4CreateTagList(MFW_GST_MP4DEMUX_INFO_T *demuxer)
{

	gchar *tag_name = NULL;
	guint count = 0;
	gchar  *codec_name = NULL;
    gint audio_track, video_track;
	GstTagList *list_tag = gst_tag_list_new();
	   

	/* sets the value for the decoder info tags */

	tag_name = GST_TAG_DURATION;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name, 
		            (guint64)demuxer->usDuration*1000, NULL);

    audio_track = demuxer->audio_track_decode;
	video_track = demuxer->video_track_decode;

	if(audio_track != -1)
	{
	    codec_name = NULL;
    	if (demuxer->tracks[audio_track].audioCodec== AUDIO_MP3)
    	{

    		if(demuxer->tracks[audio_track].numChannels == MONO)
    			codec_name = "MP3,MONO";
    
    		else if(demuxer->tracks[audio_track].numChannels == STEREO)
    			codec_name = "MP3,STEREO";
    	}
		else if (demuxer->tracks[audio_track].audioCodec == AUDIO_AAC)
		{
			if(demuxer->tracks[audio_track].numChannels == MONO)
				codec_name = "AAC LC,MONO";
			
			else if(demuxer->tracks[audio_track].numChannels == STEREO)
				codec_name = "AAC LC,STEREO";
		}
    	else if (demuxer->tracks[audio_track].audioCodec == AUDIO_AMR)
    	{
			codec_name = "AMR";    	
    	}
    	else if (demuxer->tracks[audio_track].audioCodec == AUDIO_PCM)
    	{
			codec_name = "PCM";
    	}
		if(codec_name != NULL)
		{
		     tag_name = GST_TAG_AUDIO_CODEC;
		     gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			             codec_name, NULL);
		}
		tag_name = MFW_GST_TAG_SAMPLING_FREQUENCY;
		gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			             demuxer->tracks[audio_track].sampleRate, NULL);
	}

    if(video_track != -1)
    {
        codec_name = NULL;
        if(demuxer->tracks[video_track].videoCodec ==VIDEO_MPEG4)
    	{
			codec_name = "MPEG-4";

    	}
    	else if (demuxer->tracks[video_track].videoCodec == VIDEO_H264)
    	{
			codec_name = "H264";    	
    	}
    	else if (demuxer->tracks[video_track].videoCodec == VIDEO_H263)
    	{
			codec_name = "H263";    	
    	}

    	else if (demuxer->tracks[video_track].videoCodec == VIDEO_MJPG)
    	{
			codec_name = "M-JPEG";
    	}
        if(codec_name != NULL)
        {
	        tag_name = GST_TAG_VIDEO_CODEC;
	        gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
		                 codec_name, NULL);
        }
		tag_name = MFW_GST_TAG_WIDTH;
		gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 demuxer->tracks[video_track].width, NULL);
		
		tag_name = MFW_GST_TAG_HEIGHT;
		gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			  demuxer->tracks[video_track].height, NULL);
		
		tag_name = MFW_GST_TAG_FRAMERATE;
		gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			  demuxer->tracks[video_track].video_framerate, NULL);
		
		
	}	
	
	/* sets the value for the user info tags */


	mp4_demuxer_get_usr_data(demuxer);


	if (demuxer->usrdata[USER_DATA_TITLE].data_ptr) {
	tag_name = GST_TAG_TITLE;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 demuxer->usrdata[USER_DATA_TITLE].data_ptr, NULL);
	}

	if (demuxer->usrdata[USER_DATA_ARTIST].data_ptr) {
	tag_name = GST_TAG_ARTIST;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 demuxer->usrdata[USER_DATA_ARTIST].data_ptr, NULL);
	}

	if (demuxer->usrdata[USER_DATA_GENRE].data_ptr) {
	tag_name = GST_TAG_GENRE;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 demuxer->usrdata[USER_DATA_GENRE].data_ptr, NULL);
	}

	if (demuxer->usrdata[USER_DATA_CREATION_DATE].data_ptr) {
	tag_name = MFW_GST_TAG_YEAR;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 demuxer->usrdata[USER_DATA_CREATION_DATE].data_ptr, NULL);
	}

	if (demuxer->usrdata[USER_DATA_COMMENTS].data_ptr) {
	tag_name = GST_TAG_COMMENT;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 demuxer->usrdata[USER_DATA_COMMENTS].data_ptr, NULL);
	}
	
	gst_element_found_tags(GST_ELEMENT(demuxer), list_tag);

	return;
}
/*=============================================================================
FUNCTION:            mfw_gst_mp4demuxer_change_state
    
DESCRIPTION:         This function keeps track of different states of pipeline.

ARGUMENTS PASSED:
      element    :   pointer to the mp4 demuxer element.
      transition :   state of the pipeline.
     
RETURN VALUE       
        GST_STATE_CHANGE_FAILURE    - the state change failed
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded
        GST_STATE_CHANGE_ASYNC      - the state change will happen
                                        asynchronously
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled
        
        
=============================================================================*/
static void mfw_gst_mp4_demuxer_var_init(MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer)
{
	mp4_demuxer->file_info.length = 0;
	mp4_demuxer->file_info.offset = 0;
	mp4_demuxer->file_info.buf_offset = 0;
	mp4_demuxer->segment.last_stop = 0;
	mp4_demuxer->new_seg_flag_audio = FALSE;
	mp4_demuxer->new_seg_flag_video = FALSE;
	mp4_demuxer->stop_request = FALSE;
	mp4_demuxer->buf_size = 0;
	mp4_demuxer->tmpbuf = NULL;
	mp4_demuxer->inbuff = NULL;
	mp4_demuxer->seek_flag = FALSE;
	mp4_demuxer->eos_flag[0] = FALSE;
	mp4_demuxer->eos_flag[1] = FALSE;
	mp4_demuxer->do_seek_flag = FALSE;
	mp4_demuxer->videosent = FALSE;
	
	mp4_demuxer->stream.file_handle = NULL;
	mp4_demuxer->stream.file_path = NULL;
	mp4_demuxer->parserHandle = NULL;
	mp4_demuxer->context = NULL;
}


static GstStateChangeReturn
mfw_gst_mp4demuxer_change_state(GstElement * element, GstStateChange transition)
{
    GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
    MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer = MFW_GST_MP4_DEMUXER(element);

    /***** UPWARDS *****/
    switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
    {
        GST_DEBUG
        ("\nCame to Demuxer GST_STATE_CHANGE_NULL_TO_READY\n");

		MM_INIT_DBG_MEM("mp4demux");

		mfw_gst_mp4_demuxer_var_init(mp4_demuxer);

		
        break;
    }

    case GST_STATE_CHANGE_READY_TO_PAUSED:
    {

        GST_DEBUG
        ("\nCame to Demuxer GST_STATE_CHANGE_READY_TO_PAUSED\n");
       
        /* Registers a new tag type for the use with GStreamer's type system */
        gst_tag_register (MFW_GST_TAG_WIDTH, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "image width","image width(pixel)", NULL);
        gst_tag_register (MFW_GST_TAG_HEIGHT, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "image height","image height(pixel)", NULL); 
        gst_tag_register (MFW_GST_TAG_FRAMERATE, GST_TAG_FLAG_DECODED,G_TYPE_FLOAT,
            "video framerate","number of video frames in a second", NULL);
        gst_tag_register (MFW_GST_TAG_SAMPLING_FREQUENCY, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "sampling frequency","number of audio samples per frame per second", NULL);
        gst_tag_register (MFW_GST_TAG_YEAR, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "year","year of creation", NULL);
		
        /* Initialize segment to its default values. */
        gst_segment_init(&mp4_demuxer->segment, GST_FORMAT_UNDEFINED);


        /* Sets the given activate function for the pad. The activate function will
           dispatch to activate_push or activate_pull to perform the actual activation. */

        gst_pad_set_activate_function(mp4_demuxer->sinkpad,
                      mfw_gst_mp4demuxer_activate);

        /* Sets the given activate_pull function for the pad. An activate_pull
           function prepares the element and any upstream connections for pulling. */

        gst_pad_set_activatepull_function(mp4_demuxer->sinkpad,
                          GST_DEBUG_FUNCPTR
                          (mfw_gst_mp4demuxer_activate_pull));
        break;
    }

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
    {

        GST_DEBUG
        ("\nCame to Demuxer GST_STATE_CHANGE_PAUSED_TO_PLAYING\n");
        break;
    }

    default:
    break;

    }

    
    ret = parent_class->change_state(element, transition);
    


     /***** DOWNWARDS *****/
    switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
    {
        break;
    }
    case GST_STATE_CHANGE_PAUSED_TO_READY:
    {		
        /* received stop request,do the following */
        mfw_gst_mp4_demuxer_close(mp4_demuxer);
	   
        /* Initialize segment to its default values. */
        gst_segment_init(&mp4_demuxer->segment, GST_FORMAT_UNDEFINED);

		mfw_gst_mp4_demuxer_var_init(mp4_demuxer);
        break;

    }

    case GST_STATE_CHANGE_READY_TO_NULL:
    {
		MM_DEINIT_DBG_MEM();
        break;
    }

    default:
    break;
    }

    return ret;
}

/*=============================================================================
FUNCTION:       mfw_gst_mp4demuxer_seek

DESCRIPTION:    This function is called when seek event occurs. 

ARGUMENTS PASSED:

RETURN VALUE:
         TRUE      -    if operation was successful
         FALSE     -    if operation was unsuccessful
=============================================================================*/

static gboolean
mfw_gst_mp4demuxer_seek(MFW_GST_MP4DEMUX_INFO_T* demuxer_info, GstEvent * event)
{

    GstSeekFlags flags;
    GstSeekType  cur_type;
    GstSeekType  stop_type;
    gint64       cur, curinus;
    gint64       newsegtime;
    GstFormat    format;
    gdouble      rate;
    gint64       stop;
    guint        track_count;
	gint         track_video, track_audio;
    MP4_PARSER_ERROR_CODE       Err;
	gboolean     update;
	update = FALSE;

    /* parse the event */
    gst_event_parse_seek (event, &rate, &format, &flags, &cur_type, &cur,
                          &stop_type, &stop);
    /* cur's format is ns, but our mp4 seek is us, so change it */
    /* if the seek format is in time */
    if(format == GST_FORMAT_TIME) {
        curinus = GST_TIME_AS_USECONDS(cur);

        GST_DEBUG("received seek Time : %" GST_TIME_FORMAT  " \r",GST_TIME_ARGS (cur));
        track_video = -1;
        track_audio = -1;
        for(track_count = 0; track_count < demuxer_info->total_src_pad; track_count++){	  
            if (demuxer_info->srcpadnum2trackindex[track_count].trackType == MEDIA_VIDEO)
                track_video = track_count;
            if (demuxer_info->srcpadnum2trackindex[track_count].trackType == MEDIA_AUDIO)
                track_audio = track_count;
        }

        if(track_video != -1){
            /* video track exist, so seek it. */
            Err = MP4Seek(demuxer_info->parserHandle,
                			demuxer_info->srcpadnum2trackindex[track_video].trackindex,
                			&curinus,
                			SEEK_FLAG_NO_LATER);

            if(Err != PARSER_SUCCESS)
                return FALSE;

            /* let's set segment time for segment, if accurate seek enable, so set 
            * the seek event time for it, if not enable, set seek cur time for it */
            if(demuxer_info->accurate_seek == NULL)
                newsegtime = curinus*GST_USECOND;
            else
                newsegtime = cur;

            gst_segment_set_seek(&(demuxer_info->segment), 
            				   rate,
            				   format,
            				   flags,
            				   cur_type,
            				   newsegtime,
            				   stop_type,
            				   stop,
            				   &update);
        }
              
        if(track_audio != -1) {    
            if (demuxer_info->accurate_seek){
                curinus = GST_TIME_AS_USECONDS(cur);
            }
            /* audio track exist, so continue seek audio */
            Err = MP4Seek(demuxer_info->parserHandle,
                demuxer_info->srcpadnum2trackindex[track_audio].trackindex,
                &curinus,
                SEEK_FLAG_NO_LATER);

            if(Err != PARSER_SUCCESS)
                return FALSE;	

            if (track_video == -1){/* audio only */
                if(demuxer_info->accurate_seek == NULL)
                    newsegtime = curinus * GST_USECOND;
                else
                    newsegtime = cur;

                gst_segment_set_seek(&(demuxer_info->segment), 
                                        rate,
                                        format,
                                        flags,
                                        cur_type,
                                        newsegtime,
                                        stop_type,
                                        stop,
                                        &update);
            }
        }

    }else{
		return FALSE;
    }
    
    return TRUE;
 }
 

/*=============================================================================
FUNCTION:            mfw_gst_mp4_demuxer_close
        
DESCRIPTION:         This function  Frees All Memories Allocated and Resourses. 

ARGUMENTS PASSED:    None
        
=============================================================================*/

void mfw_gst_mp4_demuxer_close(MFW_GST_MP4DEMUX_INFO_T *mp4_demuxer)
{
    guint32 stream_count;
    
    //g_print("mfw_gst_mp4_demuxer_close has been called\n");
    GST_DEBUG("Freeing all the allocated memories()\n");
    if (mp4_demuxer->tmpbuf != NULL) 
	{
        gst_buffer_unref(mp4_demuxer->tmpbuf);
        mp4_demuxer->tmpbuf=NULL;

    }

    for (stream_count=0; stream_count<MAX_SRC_PADS; stream_count++){
        MP4PadCache * cache = &mp4_demuxer->padcache[stream_count];
        if (cache->gstbuf){
            gst_buffer_unref(cache->gstbuf);
            cache->gstbuf = NULL;
            cache->offset = 0;
        }
    }
    
	for(stream_count = 0; stream_count < MP4_USRDATA_NUM; stream_count++)
	{
        if(mp4_demuxer->usrdata[stream_count].data_ptr != NULL)
			app_MP4LocalFree(mp4_demuxer->usrdata[stream_count].data_ptr);
	}
	for(stream_count = 0; stream_count < MAX_TRACK_NUM; stream_count++)
	{
        if(mp4_demuxer->tracks[stream_count].decoderSpecificInfo != NULL)
        {
			gst_buffer_unref(mp4_demuxer->tracks[stream_count].decoderSpecificInfo);
		    mp4_demuxer->tracks[stream_count].decoderSpecificInfo = NULL;
        }
	}
		
	if(mp4_demuxer->parserHandle)
	{
	    MP4DeleteParser(mp4_demuxer->parserHandle);
		mp4_demuxer->parserHandle = NULL;
	}
	return;

}

/*=============================================================================
FUNCTION:            mfw_gst_type_mp4_demuxer_get_type
        
DESCRIPTION:         intefaces are initiated in this function.you can register one
                     or more interfaces  after hmp4ng registered the type itself.


ARGUMENTS PASSED:    None
        
  
RETURN VALUE:        A numerical value ,which represents the unique identifier of this
                     element(mp4demuxer)

=============================================================================*/

GType mfw_gst_type_mp4_demuxer_get_type(void)
{
    static GType mp4demuxer_type = 0;

    if (!mp4demuxer_type) 
	{
        static const GTypeInfo mp4demuxer_info = 
		{
            sizeof(MFW_GST_MP4DEMUX_INFO_CLASS_T),
            (GBaseInitFunc) mfw_gst_mp4demuxer_base_init,
            NULL,
            (GClassInitFunc) mfw_gst_mp4_demuxer_class_init,
            NULL,
            NULL,
            sizeof(MFW_GST_MP4DEMUX_INFO_T),
            0,
            (GInstanceInitFunc) mfw_gst_mp4_demuxer_init,
        };
        mp4demuxer_type = g_type_register_static(GST_TYPE_ELEMENT,
                         "MFW_GST_MP4DEMUX_INFO_T",
                         &mp4demuxer_info, 0);
        GST_DEBUG_CATEGORY_INIT(mfw_gst_mp4demuxer_debug, "mfw_mp4demuxer",
                0, "mp4 demuxer");
    }
    return mp4demuxer_type;
}

/*=============================================================================
FUNCTION:            mfw_gst_mp4_demuxer_set_caps
         
DESCRIPTION:         this function handles the link with other plug-ins and used for
                     capability negotiation  between pads  

ARGUMENTS PASSED:    
        pad      -   pointer to GstPad
        caps     -   pointer to GstCaps

=============================================================================*/

static gboolean mfw_gst_mp4_demuxer_set_caps(GstPad * pad, GstCaps * caps)
{

    return TRUE;
}



/*=============================================================================
FUNCTION:            mfw_gst_mp4demuxer_base_init    
        
DESCRIPTION:         mp4demuxer element details are registered with the plugin during
                     _base_init ,This function will initialise the class and child
                     class properties during each new child class creation

ARGUMENTS PASSED:
        Klass   -    pointer to mp4demuxer plug-in class
  
=============================================================================*/

static void mfw_gst_mp4demuxer_base_init(MFW_GST_MP4DEMUX_INFO_CLASS_T *
					 klass)
{

    static GstElementDetails mfw_gst_mp4demuxer_details =
	GST_ELEMENT_DETAILS("FSL MP4 Demuxer",
			    "Codec/Demuxer",
			    "Demuxes an MP4 file into Audio and Video files",
			    FSL_GST_MM_PLUGIN_AUTHOR);

    GstElementClass *element_class = GST_ELEMENT_CLASS(klass);
    GST_DEBUG("Into mfw_gst_mp4demuxer_base_init( ) function \n");
    /* Create audio source pad template */

    gst_element_class_add_pad_template(element_class, audio_src_templ());
    gst_element_class_add_pad_template(element_class, video_src_templ());


    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&mfw_gst_mp4demuxer_sink_template_factory));

    gst_element_class_set_details(element_class,
				  &mfw_gst_mp4demuxer_details);

    GST_DEBUG("Out of mfw_gst_mp4demuxer_base_init function \n");
    return;
}




/*=============================================================================
FUNCTION:            mfw_gst_mp4_demuxer_class_init
        
DESCRIPTION:         Initialise the class only once (specifying what signals,
                     arguments and virtual functions the class has and setting up
                     global state)

ARGUMENTS PASSED:    klass   - pointer to mp4demuxer element class
        
=============================================================================*/

static void
mfw_gst_mp4_demuxer_class_init(MFW_GST_MP4DEMUX_INFO_CLASS_T * klass)
{
    GObjectClass *gobject_class = NULL;
    GstElementClass *gstelement_class = NULL;

    GST_DEBUG("Initialise the demuxer class\n");

    gobject_class = (GObjectClass *) klass;
    gstelement_class = (GstElementClass *) klass;
    parent_class = g_type_class_peek_parent(klass);

    gstelement_class->change_state = mfw_gst_mp4demuxer_change_state;
    gobject_class->finalize = GST_DEBUG_FUNCPTR (mfw_gst_mp4_demuxer_finalize);

	gobject_class->set_property = mfw_gst_mp4demuxer_set_property;
	gobject_class->get_property = mfw_gst_mp4demuxer_get_property;
	
	
	g_object_class_install_property(gobject_class, ID_DECODE_AUDIO_TRACK,
					g_param_spec_int("audio_track_select",
							 "audio_track_select",
							 "audio track decode from 0 to 64",
                             /* if not set or set invalid track number, default value is 0, means decode the first find audio track */
							 0, 64, 0,              
							 G_PARAM_READWRITE));
	
	g_object_class_install_property(gobject_class, ID_DECODE_VIDEO_TRACK,
					g_param_spec_int("video_track_select",
							 "video_track_select",
							 "video track decode from 0 to 64",
							 0, 64, 0,
							 G_PARAM_READWRITE));

    return;
}

/*=============================================================================
FUNCTION:            mfw_gst_mp4_demuxer_init
        
DESCRIPTION:         This function creates the pads on the elements and register the
                     function pointers which operate on these pads. 

ARGUMENTS PASSED:    
      demuxer_info  :pointer to the mp4demuxer plugin structure.
      gclass        :pointer to mp4demuxer element class.
  
RETURN VALUE:        None

=============================================================================*/
static void
mfw_gst_mp4_demuxer_init(MFW_GST_MP4DEMUX_INFO_T * demuxer_info,
             MFW_GST_MP4DEMUX_INFO_CLASS_T * gclass)
{

    gchar *padname = NULL;
    GstCaps *caps = NULL;
    GstPadTemplate *templ = NULL;
    GstPad *pad = NULL;
    gboolean set;


    GstElementClass *klass = GST_ELEMENT_GET_CLASS(demuxer_info);
    

    demuxer_info->sinkpad =
    gst_pad_new_from_template(gst_element_class_get_pad_template
                  (klass, "sink"), "sink");

    GST_DEBUG("Register the function pointers on to the sink pad\n");

    gst_pad_set_setcaps_function(demuxer_info->sinkpad,
                 mfw_gst_mp4_demuxer_set_caps);

    gst_element_add_pad(GST_ELEMENT(demuxer_info), demuxer_info->sinkpad);

    demuxer_info->audio_track_decode = 0;
	demuxer_info->video_track_decode = 0;

    gst_pad_set_event_function(demuxer_info->sinkpad,
                   GST_DEBUG_FUNCPTR
                   (mfw_gst_mp4demuxer_sink_event));

#define MFW_GST_MP4_PARSER_PLUGIN VERSION
    PRINT_CORE_VERSION(MP4ParserVersionInfo());
    PRINT_PLUGIN_VERSION(MFW_GST_MP4_PARSER_PLUGIN);
    demuxer_info->media_file_lock= g_mutex_new();

    return;
}

static void
mfw_gst_mp4_demuxer_finalize(GObject * object)
{
    MFW_GST_MP4DEMUX_INFO_T *demuxer_info;
    //g_print("mfw_gst_mp4_demuxer_finalize has been called\n");
    demuxer_info = MFW_GST_MP4_DEMUXER (object);
    g_mutex_free (demuxer_info->media_file_lock);

}
/*=============================================================================
FUNCTION:            plugin_init
        
ESCRIPTION:          special function , which is called as soon as the plugin or
                     element is loaded and information returned by this function
                     will be cached in central registry


ARGUMENTS PASSED:    
          plugin -   pointer to container that contains features loaded
                     from shared object module
        
  
RETURN VALUE:        return TRUE or FALSE depending on whether it loaded initialized any
                     dependency correctly
        
=============================================================================*/


static gboolean plugin_init(GstPlugin * plugin)
{
    return gst_element_register(plugin, "mfw_mp4demuxer",
                FSL_GST_RANK_HIGH,
                MFW_GST_TYPE_MP4_DEMUXER);
}

/*   This is used to define the entry point and meta data of plugin   */

GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,    /* major version of Gstreamer  */
          GST_VERSION_MINOR,    /* minor version of Gstreamer  */
          "mfw_mp4demuxer",     /* name of the plugin          */
          "Demuxes audio and video streams from mp4 file",  /* what plugin actually does   */
          plugin_init,          /* first function to be called */
          VERSION,
          GST_LICENSE_UNKNOWN,
          FSL_GST_MM_PLUGIN_PACKAGE_NAME, FSL_GST_MM_PLUGIN_PACKAGE_ORIG)

