/*
 * Copyright (C) 2008-2009 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_rmdemuxer.c
 *
 * Description:    RealMedia demuxer plugin for Gstreamer.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 * Aug 22 2008 Sario HU <b01138@freescale.com>
 * - Initial version
 *
 */


/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <string.h>
#include <gst/gst.h>

#include "rm_parser/rm_parser_interface.h"

#include "mfw_gst_utils.h"

#ifdef MEMORY_DEBUG
#include "mfw_gst_debug.h"
#endif

#include "mfw_gst_rmdemuxer.h"

/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/

/* None */

/*=============================================================================
                           STATIC VARIABLES
=============================================================================*/

static GstStaticPadTemplate mfw_gst_rm_demuxer_sink_factory =
GST_STATIC_PAD_TEMPLATE("sink",
			GST_PAD_SINK,
			GST_PAD_ALWAYS,
			GST_STATIC_CAPS("application/vnd.rn-realmedia")
					
    );

/*=============================================================================
                            LOCAL MACROS
=============================================================================*/

#define RM_DEMUXER_FATAL_ERROR(...) g_print(RED_STR(__VA_ARGS__))
//#define RM_PARSER_MEM_CB g_print
#define RM_DEMUXER_FLOW(...) g_print(CYAN_STR(__VA_ARGS__)) 
//#define RM_PARSER_OUTPUT_CB g_print
//#define RM_PARSER_FILE_CB g_print

    
#ifndef RM_PARSER_MEM_CB
#define RM_PARSER_MEM_CB(...)
#endif
    
#ifndef RM_DEMUXER_FLOW
#define RM_DEMUXER_FLOW(...)
#endif

#ifndef RM_PARSER_OUTPUT_CB
#define RM_PARSER_OUTPUT_CB(...)
#endif

#ifndef RM_PARSER_FILE_CB
#define RM_PARSER_FILE_CB(...)
#endif


#define MFW_GST_RM_VIDEO_CAPS \
        "video/x-pn-realvideo" 


        
/* define a tag name */
#define	MFW_GST_TAG_WIDTH		        "width"
#define MFW_GST_TAG_HEIGHT	            "height"
#define MFW_GST_TAG_FRAMERATE           "framerate"
#define MFW_GST_TAG_SAMPLING_FREQUENCY  "sampling frequency"
#define MFW_GST_TAG_YEAR                "year"


#define RM_STREAM_AUDIO_ON  (1<<0)
#define RM_STREAM_VIDEO_ON  (1<<1)

#define RM_INIT_HEAD_SIZE   16

#define RM_FOURCC_RA8LOW    0x636f6f6b
#define RM_FOURCC_ATRC      0x61747263
#define RM_FOURCC_RAVOICE   0x73697072
#define RM_FOURCC_AAC       0x72616163
#define RM_FOURCC_RACP      0x72616370

#ifdef MEMORY_DEBUG
#define RM_MEM_PRIVATE_OBJECT(dem) ((void *)(&(dem)->memmgr))
#else
#define RM_MEM_PRIVATE_OBJECT(dem) NULL
#endif

#define RMTIME2GSTTIME(rmtime) ((guint64)(rmtime)*1000000)
#define GSTTIME2RMTIME(gsttime) ((gsttime)/1000000)

#define RM_FRAMERATE_1(framerate)\
    (((guint32)(framerate))>>16)
    
#define RM_FRAMERATE(framerate)\
    ((((guint32)(framerate))>0xffff) ? RM_FRAMERATE_1(framerate) : (framerate))

#define FOURCCTO4CHAR(fourcc) ((unsigned char)(((guint32)((fourcc)))>>24)), \
                              ((unsigned char)(((guint32)((fourcc)))<<8>>24)), \
                              ((unsigned char)(((guint32)((fourcc)))<<16>>24)), \
                              ((unsigned char)(((guint32)((fourcc)))<<24>>24))

#define RM_PACK32 rm_pack32
#define RM_PACK16 rm_pack16

#define SYNCER_INIT(sync)\
    do{\
        (sync).delayhead = (sync).delaytail = NULL;\
    }while(0)

#if 0
#define DROP_VIDEO_FRAME(gstbuf, pad)\
    do{\
        GST_BUFFER_TIMESTAMP((gstbuf)) = 0;\
        gst_pad_push((pad), (gstbuf));\
    }while(0)
#else
#define DROP_VIDEO_FRAME(gstbuf, pad)\
    do{\
        g_print(BLUE_STR("video frame dropped\n", 0));\
        gst_buffer_unref(gstbuf);\
    }while(0)
#endif

//sync limitation in ns
#ifndef _MX51
#define SYNC_LIMITATION (500000000)
#else
#define  SYNC_LIMITATION (5000000000)
#endif


#define DEMUX_WAIT_INTERVAL 10000

#define MAX_TIMESTATMP(stamp1, stamp2) (((guint64)(stamp1))>((guint64)(stamp2))? ((guint64)(stamp1)):((guint64)(stamp2)))


#define TIMESTAMP_DIFF_SMALLER(stampv, stampa) \
    (((stampv)>=(stampa)) || ((stampa)-(stampv)<SYNC_LIMITATION))
    
#if 0
#define GSTBUFFERPUSHTOPAD(pad, buffer)\
    do {\
        if (((pad)==demuxer->srcpad[VIDEO_INDEX]) &&(cnt1)){\
            g_print("push %s %d tm %"GST_TIME_FORMAT" %s\n", "video", cnt1,  GST_TIME_ARGS(GST_BUFFER_TIMESTAMP(buffer)), (GST_BUFFER_FLAG_IS_SET(buffer,GST_BUFFER_FLAG_LAST))? "key":"non-key");\
            cnt1--;\
        }\
        if (((pad)==demuxer->srcpad[AUDIO_INDEX]) &&(cnt2)){\
            g_print("push %s %d tm %"GST_TIME_FORMAT" %s\n", "audio", cnt2, GST_TIME_ARGS(GST_BUFFER_TIMESTAMP(buffer)), (GST_BUFFER_FLAG_IS_SET(buffer,GST_BUFFER_FLAG_LAST))? "key":"non-key");\
            cnt2--;\
        }\
        gst_pad_push((pad), (buffer));\
    }while(0)
#else

#define GSTBUFFERPUSHTOPAD(pad, buffer)\
            do {    GstFlowReturn ret=GST_FLOW_OK;\
                ret=gst_pad_push((pad), (buffer));\
                if (ret != GST_FLOW_OK)  GST_ERROR("Push data failed:%d",ret);\
            }while(0)

#endif

#define SET_VIDEO_DROP(demuxer)\
    do{\
        g_print(BLUE_STR("set video drop \n", 0));\       
        (demuxer)->videodropto=TRUE;\
    }while(0)

/*=============================================================================
                           LOCAL VARIABLES
=============================================================================*/

/* None. */

/*=============================================================================
                        STATIC FUNCTION PROTOTYPES
=============================================================================*/

GST_DEBUG_CATEGORY_STATIC(mfw_gst_rmdemuxer_debug);
static void mfw_gst_rm_demuxer_class_init(MFW_GST_RMDEMUX_INFO_CLASS_T
					   *);
static void mfw_gst_rm_demuxer_base_init(MFW_GST_RMDEMUX_INFO_CLASS_T *);
static void mfw_gst_rm_demuxer_init(MFW_GST_RMDEMUX_INFO_T *,
				     MFW_GST_RMDEMUX_INFO_CLASS_T *);
static gboolean mfw_gst_rm_demuxer_set_caps(GstPad *, GstCaps *);
static GstStateChangeReturn mfw_gst_rmdemuxer_change_state(GstElement *,
							    GstStateChange);
static gboolean mfw_gst_rmdemuxer_activate(GstPad *);
static gboolean mfw_gst_rmdemuxer_activate_pull(GstPad *, gboolean);
static gboolean mfw_gst_rmdemuxer_sink_event(GstPad *, GstEvent *);

static GstPadTemplate *audio_src_templ(void);
static GstPadTemplate *video_src_templ(void);
static gboolean mfw_gst_rmdemuxer_fill_stream_info(MFW_GST_RMDEMUX_INFO_T *);
static gboolean rm_demuxer_set_pad_caps(MFW_GST_RMDEMUX_INFO_T *);
static gboolean mfw_gst_rmdemuxer_parse(MFW_GST_RMDEMUX_INFO_T *);
static gboolean mfw_gst_rmdemuxer_handle_src_query(GstPad *,
						    GstQuery *);

static void mfw_gst_rm_demuxer_close(MFW_GST_RMDEMUX_INFO_T *);
static const GstQueryType *gst_rmdemuxer_get_src_query_types(GstPad *);
							      
static gboolean mfw_gst_rmdemuxer_handle_src_event(GstPad*,GstEvent*);

/*=============================================================================
                            STATIC VARIABLES
=============================================================================*/

static GstElementClass *parent_class = NULL;


static 
void RMError(void* pError, HX_RESULT result, const char* pszMsg)
{
    RM_DEMUXER_FATAL_ERROR("%s err:%p, result:%ld, %s\n", __FUNCTION__, pError, result, pszMsg);
}

static
void* RMMalloc(void* pUserMem, UINT32 ulSize)
{
    RM_PARSER_MEM_CB("%s:%p alloc %ld\n", __FUNCTION__, pUserMem, ulSize);
#ifdef MEMORY_DEBUG
    return dbg_malloc(pUserMem, ulSize, NULL);
#else
    return g_malloc(ulSize);
#endif
}

static
void  RMFree(void* pUserMem, void* ptr)
{
    RM_PARSER_MEM_CB("%s:%p free %p\n", __FUNCTION__, pUserMem, ptr);
#ifdef MEMORY_DEBUG
    dbg_free(pUserMem, ptr, NULL);
#else
    g_free(ptr);
#endif
}

static
UINT32 RMRead(void* pUserRead, BYTE* pBuf, UINT32 ulBytesToRead)
{
    MFW_GST_RMDEMUX_INFO_T * demuxer = (MFW_GST_RMDEMUX_INFO_T *)pUserRead;
    GstBuffer * gstbuf;
    UINT32 ret = 0;
    if (gst_pad_pull_range(demuxer->sinkpad, demuxer->file_info.offset,
                            ulBytesToRead, &gstbuf) == GST_FLOW_OK){
        ret = GST_BUFFER_SIZE(gstbuf);
        demuxer->file_info.offset+=ret;
        memcpy(pBuf, GST_BUFFER_DATA(gstbuf), ret);
        gst_buffer_unref(gstbuf);
        demuxer->readerror = FALSE;
    }else{
        demuxer->readerror = TRUE;
        RM_DEMUXER_FATAL_ERROR("error while read\n", 0);
        return ret;
    }
    RM_PARSER_FILE_CB("%s read %ld actual %ld\n", __FUNCTION__, ulBytesToRead, ret);
    return ret;
}



static
void RMSeek (void* pUserRead, UINT32 ulOffset, UINT32 ulOrigin)
{
    MFW_GST_RMDEMUX_INFO_T * demuxer = (MFW_GST_RMDEMUX_INFO_T *)pUserRead;
    guint64 bases;
    switch(ulOrigin){
        case 0:
        bases = 0;
        break;
        case 1:
        bases = demuxer->file_info.offset;
        break;
        case 2:
        bases = demuxer->file_info.length;
        break;
        default:
        return;
    };
    demuxer->file_info.offset = bases+ulOffset;
    RM_PARSER_FILE_CB("%s seek orig %ld to %ld\n", __FUNCTION__, ulOrigin, demuxer->file_info.offset);
}


static void mfw_gst_rmdemuxer_send_newsegment_all(MFW_GST_RMDEMUX_INFO_T * demuxer)
{
    int i;
    GstPad * srcpad;
    guint64 new_seg_time = demuxer->actual_seek_time;
    FLAG_CLEAR_BIT(demuxer->new_seg_flag, (SEGMENT_NOT_SEND_MASK|FIRST_MASK));
    RM_DEMUXER_FLOW("try to send newsegment %"GST_TIME_FORMAT" video%"GST_TIME_FORMAT" audio%"GST_TIME_FORMAT".\n"
        , GST_TIME_ARGS(new_seg_time), GST_TIME_ARGS(demuxer->firstvideo_time), GST_TIME_ARGS(demuxer->firstaudio_time));
    for (i=0;i<MAX_SRC_PADS;i++){
        if (srcpad = demuxer->srcpad[i]){
            if (!gst_pad_push_event(srcpad,
                    gst_event_new_new_segment(FALSE, 1.0, GST_FORMAT_TIME, 
                    new_seg_time,
                    GST_CLOCK_TIME_NONE, new_seg_time))) {
                        RM_DEMUXER_FATAL_ERROR("Cannot send new segment to the %s pad!\n", (i==AUDIO_INDEX) ? "audio" : "video");
	            }
        }
    }
}

static void mfw_gst_process_a_delay(MFW_GST_RMDEMUX_INFO_T * demuxer)
{
    GstBuffer * gstbuf;
    GstPad * pad = demuxer->srcpad[AUDIO_INDEX]; 
    SyncControl * pcontrol = &demuxer->audiosync;


    while (gstbuf=pcontrol->delayhead)
    {
        pcontrol->delayhead=GST_BUFFER_OFFSET_END(gstbuf);
        if (GST_BUFFER_TIMESTAMP(gstbuf)>=demuxer->actual_seek_time)
            GSTBUFFERPUSHTOPAD(pad, gstbuf);
        else
            gst_buffer_unref(gstbuf);
    }
    pcontrol->delayhead = pcontrol->delaytail=NULL;
}


static void writeAudioFrame(MFW_GST_RMDEMUX_INFO_T * demuxer, GstBuffer * gstbuf)
{
    if (GST_BUFFER_TIMESTAMP(gstbuf)>=demuxer->actual_seek_time){
        GSTBUFFERPUSHTOPAD(demuxer->srcpad[AUDIO_INDEX], gstbuf);
    }else{
        g_print(BLUE_STR("drop audio block \n", 0));
        gst_buffer_unref(gstbuf);
    }
}

static void writeVideoFrame(MFW_GST_RMDEMUX_INFO_T * demuxer, GstBuffer * buffer)
{
    GstFlowReturn ret;
    GstPad * pad = demuxer->srcpad[VIDEO_INDEX];
        
    if (G_UNLIKELY(demuxer->seek_video==TRUE)){
        rv_format_info * rvinfo = demuxer->rvinfo;

        if (rvinfo){
            gint size;
            GstBuffer * tmpbuf;
            GstClockTime ts = GST_BUFFER_TIMESTAMP(buffer);
            
            size = rvinfo->ulLength;
            if (gst_pad_alloc_buffer(pad, 0, size, NULL,&tmpbuf)
                    ==GST_FLOW_OK){
                PackRVFormatInfo(rvinfo, GST_BUFFER_DATA(tmpbuf), size);
            }
            buffer = gst_buffer_join(tmpbuf,buffer);
            gst_buffer_set_caps(buffer,GST_PAD_CAPS(pad));

            GST_BUFFER_TIMESTAMP(buffer) = ts;
            
        }
        demuxer->seek_video = FALSE;
    }

    ret = gst_pad_push(pad, buffer);
    if (ret != GST_FLOW_OK){
        GST_ERROR("Push data failed:%d",ret);
    }

    demuxer->videosent = TRUE;
    
}



static void mfw_gst_q_delay(SyncControl* pcontrol, GstBuffer * gstbuf )
{
    GST_BUFFER_OFFSET_END(gstbuf) = 0;

        if (pcontrol->delaytail){
            GST_BUFFER_OFFSET_END(pcontrol->delaytail) = gstbuf;
            pcontrol->delaytail = gstbuf;
        }else{
            pcontrol->delaytail=pcontrol->delayhead = gstbuf;
        }
    
}

static void mfw_gst_process_v_delay(MFW_GST_RMDEMUX_INFO_T * demuxer)
{
    GstBuffer * gstbuf;
    GstPad * pad = demuxer->srcpad[VIDEO_INDEX]; 
    SyncControl * pcontrol = &demuxer->videosync;
    
    if (demuxer->videodropto){
        while (gstbuf=pcontrol->delayhead){
            
            if ((GST_BUFFER_FLAG_IS_SET(gstbuf,GST_BUFFER_FLAG_LAST))
                && (TIMESTAMP_DIFF_SMALLER(GST_BUFFER_TIMESTAMP(gstbuf), demuxer->firstaudio_time))){
                    demuxer->videodropto = FALSE;
                    demuxer->videokeyfound = TRUE;
                    demuxer->actual_seek_time = MAX_TIMESTATMP(GST_BUFFER_TIMESTAMP(gstbuf), demuxer->firstaudio_time);
                    mfw_gst_rmdemuxer_send_newsegment_all(demuxer);
                    mfw_gst_process_a_delay(demuxer);
                    break;
            }else{
                    pcontrol->delayhead=GST_BUFFER_OFFSET_END(gstbuf);
                    DROP_VIDEO_FRAME(gstbuf, pad);
                
            }
        }
    
    }
    
    while (gstbuf=pcontrol->delayhead)
    {
        pcontrol->delayhead=GST_BUFFER_OFFSET_END(gstbuf);

            writeVideoFrame(demuxer, gstbuf);
    }
    pcontrol->delayhead = pcontrol->delaytail=NULL;
}

static ReStampVideoFrame(MFW_GST_RMDEMUX_INFO_T * demuxer, guint32 rmtimestamp)
{
    int i = 0;
    GstBuffer * gstbuf = demuxer->vhead, * gstnextbuf;
    guint64 timestart = RMTIME2GSTTIME(GST_BUFFER_TIMESTAMP(gstbuf));
    rmtimestamp -= GST_BUFFER_TIMESTAMP(gstbuf);
    while(gstbuf){
        gstnextbuf = (GstBuffer *)(GST_BUFFER_OFFSET_END(gstbuf));
        GST_BUFFER_TIMESTAMP(gstbuf) = gst_util_uint64_scale(i, RMTIME2GSTTIME(rmtimestamp), demuxer->vframenum)+timestart;

        if (demuxer->videokeyfound==FALSE){
            if (!(GST_BUFFER_FLAG_IS_SET(gstbuf,GST_BUFFER_FLAG_LAST))){
                gst_buffer_unref(gstbuf);
                gstbuf = gstnextbuf;
                i++;
                printf("drop non-key frame\n");
                continue;
            }else{
                demuxer->videokeyfound = TRUE;
            }
        }

        if (FLAG_TEST_BIT(demuxer->new_seg_flag, VIDEO_MASK)){
            FLAG_CLEAR_BIT(demuxer->new_seg_flag, VIDEO_MASK);
            demuxer->firstvideo_time = GST_BUFFER_TIMESTAMP(gstbuf);

            if (!FLAG_TEST_BIT(demuxer->new_seg_flag, AUDIO_MASK)){
                if (demuxer->firstvideo_time>=demuxer->firstaudio_time){
                    demuxer->actual_seek_time = demuxer->firstvideo_time;
                    mfw_gst_rmdemuxer_send_newsegment_all(demuxer);
                    mfw_gst_process_a_delay(demuxer);
                }else if (((demuxer->firstaudio_time-demuxer->firstvideo_time)<SYNC_LIMITATION) 
                          || (FLAG_TEST_BIT(demuxer->new_seg_flag, FIRST_MASK))) {
                        demuxer->actual_seek_time = demuxer->firstaudio_time;
                        mfw_gst_rmdemuxer_send_newsegment_all(demuxer);
                        mfw_gst_process_a_delay(demuxer);
                }else{
                    SET_VIDEO_DROP(demuxer);
                    demuxer->videokeyfound = FALSE;
                }
            }

        }

        if (FLAG_TEST_BIT(demuxer->new_seg_flag, AUDIO_MASK)){
            mfw_gst_q_delay(&demuxer->videosync, gstbuf);
        }else{
            if (demuxer->videodropto){
                if (GST_BUFFER_FLAG_IS_SET(gstbuf,GST_BUFFER_FLAG_LAST)
                && (TIMESTAMP_DIFF_SMALLER(GST_BUFFER_TIMESTAMP(gstbuf), demuxer->firstaudio_time))){
                    demuxer->videodropto = FALSE;
                    demuxer->videokeyfound = TRUE;
                    demuxer->actual_seek_time = MAX_TIMESTATMP(GST_BUFFER_TIMESTAMP(gstbuf), demuxer->firstaudio_time);
                    mfw_gst_rmdemuxer_send_newsegment_all(demuxer);
                    writeVideoFrame(demuxer, gstbuf);
                    mfw_gst_process_a_delay(demuxer);
                }else{
                DROP_VIDEO_FRAME(gstbuf, demuxer->srcpad[VIDEO_INDEX]);
                }
                
            }else{
                writeVideoFrame(demuxer, gstbuf);
            }
        }
        gstbuf = gstnextbuf;
        i++;
    }
}



void PackRVFormatInfo(rv_format_info* pInfo, BYTE* pBuf, UINT32 ulLen)
{
    if (pInfo && pBuf && ulLen >= pInfo->ulLength)
    {
        RM_PACK32(pInfo->ulLength,          &pBuf, &ulLen);
        RM_PACK32(pInfo->ulMOFTag,          &pBuf, &ulLen);
        RM_PACK32(pInfo->ulSubMOFTag,       &pBuf, &ulLen);
        RM_PACK16(pInfo->usWidth,           &pBuf, &ulLen);
        RM_PACK16(pInfo->usHeight,          &pBuf, &ulLen);
        RM_PACK16(pInfo->usBitCount,        &pBuf, &ulLen);
        RM_PACK16(pInfo->usPadWidth,        &pBuf, &ulLen);
        RM_PACK16(pInfo->usPadHeight,       &pBuf, &ulLen);
        RM_PACK32(pInfo->ufFramesPerSecond, &pBuf, &ulLen);
        /* Sanity Check */
        if (ulLen >= pInfo->ulOpaqueDataSize)
        {
            memcpy(pBuf, pInfo->pOpaqueData, pInfo->ulOpaqueDataSize);
        }else{
            g_print("%s: buffer space not enough\n", __FUNCTION__);
        }
    }
}


void PackFrame(rv_frame* pFrame, char * pBuf, guint ulLen)
{
    if (pFrame && pBuf)
    {
        guint ulPackLen = 20 + pFrame->ulNumSegments * 8 + pFrame->ulDataLen;
        guint i         = 0;
        if (ulLen >= ulPackLen)
        {
            RM_PACK32(pFrame->ulDataLen,     &pBuf, &ulLen);
            RM_PACK32(pFrame->ulTimestamp,   &pBuf, &ulLen);
            RM_PACK16(pFrame->usSequenceNum, &pBuf, &ulLen);
            RM_PACK16(pFrame->usFlags,       &pBuf, &ulLen);
            RM_PACK32(pFrame->bLastPacket,   &pBuf, &ulLen);
            RM_PACK32(pFrame->ulNumSegments, &pBuf, &ulLen);
            for (i = 0; i < pFrame->ulNumSegments; i++)
            {
                RM_PACK32((guint) pFrame->pSegment[i].bIsValid, &pBuf, &ulLen);
                RM_PACK32((guint) pFrame->pSegment[i].ulOffset, &pBuf, &ulLen);
            }
            /* Sanity check */
            if (ulLen >= pFrame->ulDataLen)
            {
                memcpy((void*) pBuf, pFrame->pData, pFrame->ulDataLen);
            }else{
                g_print("%s: buffer space not enough\n", __FUNCTION__);
            }
        }
    }
}

HX_RESULT RMVideoFrameAvail(void* pAvail, UINT32 ulSubStreamNum, rv_frame* pFrame)
{
    HX_RESULT retVal = HXR_OK;
    MFW_GST_RMDEMUX_INFO_T * demuxer = (MFW_GST_RMDEMUX_INFO_T *)pAvail;
    rv_depack * rvdepack = demuxer->rvdepack;
    GstPad * pad = demuxer->srcpad[VIDEO_INDEX];
    GstBuffer * gstbuf = NULL;
    guint32 size;
    rv_format_info * rvinfo = demuxer->rvinfo;

    if (pAvail && pFrame){
        size = 20 + pFrame->ulNumSegments * 8 + pFrame->ulDataLen;

        if (gst_pad_alloc_buffer(pad, 0, size, GST_PAD_CAPS(pad),&gstbuf)==GST_FLOW_OK){
            GST_BUFFER_TIMESTAMP(gstbuf) = pFrame->ulTimestamp;
            PackFrame(pFrame, GST_BUFFER_DATA(gstbuf), size);
        }
           
        if (gstbuf){

            if (pFrame->usFlags & HX_KEYFRAME_FLAG){
                GST_BUFFER_FLAG_SET (gstbuf, GST_BUFFER_FLAG_LAST);
            }
             
            GST_BUFFER_OFFSET_END(gstbuf) = 0;

            if (demuxer->vtail){
                guint32 lasttimestamp = GST_BUFFER_TIMESTAMP(demuxer->vtail);
                if ((!(pFrame->usFlags & HX_KEYFRAME_FLAG)) 
                    || ((lasttimestamp+1)==pFrame->ulTimestamp)){
                    GST_BUFFER_OFFSET_END(demuxer->vtail) = gstbuf;
                    demuxer->vtail = gstbuf;
                    demuxer->vframenum ++;
                }else{
                    ReStampVideoFrame(demuxer, pFrame->ulTimestamp);
                    demuxer->vhead = demuxer->vtail = gstbuf;
                    demuxer->vframenum = 1;
                }
            }else{
                demuxer->vhead = demuxer->vtail = gstbuf;
                demuxer->vframenum = 1;
            }
        }
        
        RM_PARSER_OUTPUT_CB("video frame len %d\n", pFrame->ulDataLen);

        /* Clean up the frame */
        rv_depack_destroy_frame(rvdepack, &pFrame);
    }
    
    return retVal;
}

HX_RESULT RMAudioFrameAvail(void* pAvail, UINT32 ulSubStreamNum, ra_block* pBlock)
{
	HX_RESULT retVal = HXR_OK;
    GstBuffer * gstbuf = NULL;
    
    MFW_GST_RMDEMUX_INFO_T * demuxer = (MFW_GST_RMDEMUX_INFO_T *)pAvail;
    ra_depack * radepack = demuxer->radepack;
    GstPad * pad = demuxer->srcpad[AUDIO_INDEX];
    
	if (pAvail && pBlock){
        
        if (FLAG_TEST_BIT(demuxer->new_seg_flag, AUDIO_MASK)){

            if ((demuxer->audiokeyfound==FALSE) && (!(pBlock->ulDataFlags & HX_KEYFRAME_FLAG))){
                return retVal;
            }
            demuxer->audiokeyfound = TRUE;
                
            FLAG_CLEAR_BIT(demuxer->new_seg_flag, AUDIO_MASK);
            demuxer->firstaudio_time = RMTIME2GSTTIME(pBlock->ulTimestamp);

            if (!FLAG_TEST_BIT(demuxer->new_seg_flag, VIDEO_MASK)){
                if ((demuxer->firstvideo_time>=demuxer->firstaudio_time)){
                    demuxer->actual_seek_time = demuxer->firstvideo_time;
                    mfw_gst_rmdemuxer_send_newsegment_all(demuxer);
                    mfw_gst_process_v_delay(demuxer);
                }else if (((demuxer->firstaudio_time-demuxer->firstvideo_time)<SYNC_LIMITATION)
                ||(FLAG_TEST_BIT(demuxer->new_seg_flag, FIRST_MASK))){
                    demuxer->actual_seek_time = demuxer->firstaudio_time;
                    mfw_gst_rmdemuxer_send_newsegment_all(demuxer);
                    mfw_gst_process_v_delay(demuxer);
                }else if (!(demuxer->streamMask & VIDEO_MASK)){//ONLY AUDIO
                    demuxer->actual_seek_time = demuxer->firstaudio_time;
                    mfw_gst_rmdemuxer_send_newsegment_all(demuxer);
                }else{
                    SET_VIDEO_DROP(demuxer);
                    demuxer->videokeyfound = FALSE;
                    mfw_gst_process_v_delay(demuxer);
                }
            }

        }
      

        if (gst_pad_alloc_buffer(pad, 0, pBlock->ulDataLen, GST_PAD_CAPS(pad),&gstbuf)
            ==GST_FLOW_OK){
            memcpy(GST_BUFFER_DATA(gstbuf), pBlock->pData, pBlock->ulDataLen);
            GST_BUFFER_TIMESTAMP(gstbuf) = RMTIME2GSTTIME(pBlock->ulTimestamp);
                
        }


        
        if (gstbuf){
            
            if (pBlock->ulDataFlags & HX_KEYFRAME_FLAG)
                GST_BUFFER_FLAG_SET (gstbuf, GST_BUFFER_FLAG_LAST);
            
            if ((FLAG_TEST_BIT(demuxer->new_seg_flag, VIDEO_MASK))
                || (demuxer->videodropto)){
                mfw_gst_q_delay(&demuxer->audiosync, gstbuf);
            }else{
                writeAudioFrame(demuxer, gstbuf);
            }
        }
        RM_PARSER_OUTPUT_CB("audio frame len %d\n", pBlock->ulDataLen);
	}
	return retVal;
}

static void mfw_gst_clean_all_queue(MFW_GST_RMDEMUX_INFO_T * demuxer)
{
    GstBuffer * gstbuf = demuxer->vhead, * gstbufnext;

    while(gstbuf){
        gstbufnext = (GstBuffer *)(GST_BUFFER_OFFSET_END(gstbuf));
        gst_buffer_unref(gstbuf);
        gstbuf = gstbufnext;
    }


    demuxer->vhead = demuxer->vtail = NULL;
    demuxer->vframenum = 0;

    gstbuf=demuxer->videosync.delayhead;
    while(gstbuf){
        gstbufnext = (GstBuffer *)(GST_BUFFER_OFFSET_END(gstbuf));
        gst_buffer_unref(gstbuf);
        gstbuf = gstbufnext;
    }
    SYNCER_INIT(demuxer->videosync);

    gstbuf=demuxer->audiosync.delayhead;
    while(gstbuf){
        gstbufnext = (GstBuffer *)(GST_BUFFER_OFFSET_END(gstbuf));
        gst_buffer_unref(gstbuf);
        gstbuf = gstbufnext;
    }
    SYNCER_INIT(demuxer->audiosync);
    
}


static void mfw_gst_eos_cache(MFW_GST_RMDEMUX_INFO_T * demuxer)
{
    GstBuffer * gstbuf = demuxer->vhead, * gstbufnext;
    GstPad * pad = demuxer->srcpad[VIDEO_INDEX];

    if ((!FLAG_TEST_BIT(demuxer->new_seg_flag, SEGMENT_NOT_SEND_MASK )) && (gstbuf) && (pad)){
        guint64 realtimestamp = RMTIME2GSTTIME(GST_BUFFER_TIMESTAMP(gstbuf));
        guint64 timestep = 1000000000/30;
        if ((demuxer->rvinfo) && (demuxer->rvinfo->ufFramesPerSecond)){
            timestep = 1000000000/RM_FRAMERATE(demuxer->rvinfo->ufFramesPerSecond);/*ENGR00112713*/
        }
        while(gstbuf){
            gstbufnext = (GstBuffer *)(GST_BUFFER_OFFSET_END(gstbuf));
            GST_BUFFER_TIMESTAMP(gstbuf) = realtimestamp;
            writeVideoFrame(demuxer, gstbuf);

            gstbuf = gstbufnext;
            realtimestamp+=timestep;
            
        }
    }else{
        while(gstbuf){
            gstbufnext = (GstBuffer *)(GST_BUFFER_OFFSET_END(gstbuf));
            gst_buffer_unref(gstbuf);
            gstbuf = gstbufnext;
        }
    }

    demuxer->vhead = demuxer->vtail = NULL;
    demuxer->vframenum = 0;

    gstbuf=demuxer->videosync.delayhead;
    while(gstbuf){
        gstbufnext = (GstBuffer *)(GST_BUFFER_OFFSET_END(gstbuf));
        gst_buffer_unref(gstbuf);
        gstbuf = gstbufnext;
    }
    SYNCER_INIT(demuxer->videosync);

    gstbuf=demuxer->audiosync.delayhead;
    while(gstbuf){
        gstbufnext = (GstBuffer *)(GST_BUFFER_OFFSET_END(gstbuf));
        gst_buffer_unref(gstbuf);
        gstbuf = gstbufnext;
    }
    SYNCER_INIT(demuxer->audiosync);
    
}


static void mfw_gst_rmdemuxer_core_cleanup(MFW_GST_RMDEMUX_INFO_T * demuxer)
{

    if (demuxer->rainfo){
        ra_depack_destroy_codec_init_info(demuxer->radepack, &demuxer->rainfo);
        demuxer->rainfo = NULL;
    }
    if (demuxer->rvinfo){
        rv_depack_destroy_codec_init_info(demuxer->rvdepack, &demuxer->rvinfo);
        demuxer->rvinfo = NULL;
    }
    
    if (demuxer->radepack){
        ra_depack_destroy(&demuxer->radepack);
        demuxer->radepack = NULL;
    }
    if (demuxer->rvdepack){
        rv_depack_destroy(&demuxer->rvdepack);
        demuxer->rvdepack = NULL;
    }

    if (demuxer->raheader){
        rm_parser_destroy_stream_header(demuxer->rmparser, &demuxer->raheader);
        demuxer->raheader = NULL;
    }
    if (demuxer->rvheader){
        rm_parser_destroy_stream_header(demuxer->rmparser, &demuxer->rvheader);
        demuxer->rvheader = NULL;
    }

    if (demuxer->rmparser){
        rm_parser_destroy(&demuxer->rmparser);
        demuxer->rmparser = NULL;
    }
}

/*=============================================================================
FUNCTION:            mfw_gst_rmdemuxer_parse
        
DESCRIPTION:         This is the main functions, where calls to RM Parser 
					 are made.

ARGUMENTS PASSED:    demuxer - The context of the main rm demuxer element.
  
RETURN VALUE:        None

PRE-CONDITIONS:      None
 
POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/
static gboolean mfw_gst_rmdemuxer_parse(MFW_GST_RMDEMUX_INFO_T * demuxer)
{
    GstPad *src_pad;
    GstBuffer * gstbuf;
    gboolean ret_value = FALSE;
    guint track_count;		/* track counter */
    guint stream_count;		/* number of streams */
	guint64 duration;     /* stream duration */ 
	gchar *text_msg = NULL;
    rm_parser * rmparser = NULL;
    
    int i;
    int value;

    if (gst_pad_pull_range(demuxer->sinkpad, 0, RM_INIT_HEAD_SIZE, &gstbuf) == GST_FLOW_OK){
        if (GST_BUFFER_SIZE(gstbuf)==RM_INIT_HEAD_SIZE){
            if (!rm_parser_is_rm_file(GST_BUFFER_DATA(gstbuf), RM_INIT_HEAD_SIZE)){
                GST_ERROR("No valid realmedia file found\n");
                gst_buffer_unref(gstbuf);
                goto InitErr;
            }else{
            }
        }else{
            gst_buffer_unref(gstbuf);
            goto InitErr;
        }
    }else{
        GST_ERROR("can not get enough content\n");
        goto InitErr;
    }
    
    rmparser = rm_parser_create2(demuxer, RMError, RM_MEM_PRIVATE_OBJECT(demuxer), RMMalloc, RMFree);

    if (rmparser==NULL){
        GST_ERROR("Can not create parser\n");
        goto InitErr;
    }

    demuxer->rmparser = rmparser;

    if (rm_parser_init_io(rmparser, demuxer, RMRead, RMSeek)!=HXR_OK){
        GST_ERROR("init io error\n");
        goto InitErr;
    }


    if (rm_parser_read_headers(rmparser)!=HXR_OK){
        GST_ERROR("rm_parser_read_headers error\n");
        goto InitErr;
    }
    
    if (demuxer->stop_request == FALSE) {
        ret_value = mfw_gst_rmdemuxer_fill_stream_info(demuxer);
        demuxer->new_seg_flag = (demuxer->streamMask | FIRST_MASK| SEGMENT_NOT_SEND_MASK);
    }
    
    gst_segment_set_duration(&demuxer->segment, GST_FORMAT_TIME, RMTIME2GSTTIME(rm_parser_get_duration(rmparser)));
    
    for (stream_count = 0; stream_count < MAX_SRC_PADS; stream_count++) {
        if (demuxer->srcpad[stream_count]){
        	gst_pad_set_query_type_function(demuxer->srcpad[stream_count],
        					                GST_DEBUG_FUNCPTR
        					                 (gst_rmdemuxer_get_src_query_types));

        	gst_pad_set_query_function(demuxer->srcpad[stream_count],
        				                    GST_DEBUG_FUNCPTR
        				                     (mfw_gst_rmdemuxer_handle_src_query));

        	gst_pad_set_event_function(demuxer->srcpad[stream_count],
                                            GST_DEBUG_FUNCPTR (
        		                             mfw_gst_rmdemuxer_handle_src_event));
        }
    }
    
    gst_element_no_more_pads (GST_ELEMENT (demuxer));

	/* Add the user data information to the taglist */
	//RMCreateTagList(demuxer);
    return ret_value;

	/* send an error message for an unrecoverable error */
InitErr:
	{
	return FALSE;
	}
 }

static gboolean mfw_gst_rmdemuxer_try_seek_to_time(MFW_GST_RMDEMUX_INFO_T* demuxer, guint64 timetoseek)
{


    mfw_gst_clean_all_queue(demuxer);
    demuxer->firstaudio_time=demuxer->firstvideo_time=demuxer->actual_seek_time=0;
    rm_parser_seek(demuxer->rmparser, GSTTIME2RMTIME(timetoseek));
    rv_depack_seek(demuxer->rvdepack, GSTTIME2RMTIME(timetoseek));
    ra_depack_seek(demuxer->radepack, GSTTIME2RMTIME(timetoseek));
        
    return TRUE;
}


/*=============================================================================
FUNCTION:            mfw_gst_rmdemuxer_taskfunc
        
DESCRIPTION:         This is the main functions, where audio and video data are
                     Read from the file 

ARGUMENTS PASSED:    demuxer - The context of the main rm demuxer element.
  
RETURN VALUE:        None

PRE-CONDITIONS:      None
 
POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/
void mfw_gst_rmdemuxer_taskfunc(GstPad * pad)
{
    HX_RESULT retVal;
    rm_parser * pParser;
    rm_packet * pPacket = NULL;

    MFW_GST_RMDEMUX_INFO_T *demuxer =
	    MFW_GST_RM_DEMUXER(GST_PAD_PARENT(pad));

    /* Current RM demuxer architecture will collect a set of packets first,
    *  Push the data continuously in this case, it could cause the video decoder
    *  request buffer from pad while the video render is not linked.
    */
    if (demuxer->freshstart) {
        demuxer->freshstart = FALSE;
        retVal = mfw_gst_rmdemuxer_parse(demuxer);
        if(retVal==FALSE) {
            /* if the parsing failed due to unsupported stream or 
            corrupted stream,return back */
            GST_ERROR("Could not parse the streams.");   
            mfw_gst_rm_demuxer_close(demuxer);
            goto pauseTask;
        }
    }
    if (demuxer->seek_flag){  
        gboolean res;
        res = mfw_gst_rmdemuxer_try_seek_to_time(demuxer, demuxer->desired_seek_time);
        demuxer->seek_flag = FALSE;

        if(res == FALSE) {
            GST_ERROR("Failed in demuxer seek !!\n");
            return ;
        }
    }

	/* Get the next packet */
    pParser  = demuxer->rmparser;
    retVal = rm_parser_get_packet(pParser, &pPacket);
    
    if (G_LIKELY(retVal == HXR_OK)) {
        /* Is this an audio packet? */
        if (pPacket->usStream == demuxer->streamID[AUDIO_INDEX]){
        	/* Put the packet into the depacketizer */
        	retVal = ra_depack_add_packet(demuxer->radepack, pPacket);
        }
        /* Is this an video packet? */
        else if (pPacket->usStream == demuxer->streamID[VIDEO_INDEX]) {
        	/* Put the packet into the depacketizer */
        	retVal = rv_depack_add_packet(demuxer->rvdepack, pPacket);
        }else {
            GST_WARNING("Unexpect packet from stream %d\n!", pPacket->usStream);
        }
    
        /* Destroy the packet */
        rm_parser_destroy_packet(pParser, &pPacket);
        
    }else if ((retVal == HXR_AT_END) 
        || ((retVal == HXR_READ_ERROR) && (demuxer->readerror==FALSE))
        || (retVal == HXR_CORRUPT_FILE)
        || (retVal == HXR_BAD_FORMAT)){
        
        gint stream_count;
        
        RM_DEMUXER_FLOW("send eos down stream %d \n", retVal);
        mfw_gst_eos_cache(demuxer);

        for (stream_count=0; stream_count<MAX_SRC_PADS; stream_count++) {
            GstPad * src_pad = demuxer->srcpad[stream_count];
            if (src_pad){
                 /*Handle an event on the sink pad with event as End of Stream */
                if(mfw_gst_rmdemuxer_sink_event(src_pad, gst_event_new_eos())!= TRUE){
                    GST_ERROR("\n Error in pushing the event.\n");
                }
            }
        }
        goto pauseTask;

    }else{
        RM_DEMUXER_FATAL_ERROR("rm_parser_get_packet failed with return %lx\n", retVal);
    }
    
    if (demuxer->videosent) {
        /* Keep the balance of demuxer and codec */    
        usleep(DEMUX_WAIT_INTERVAL);
        demuxer->videosent = FALSE;
    }
    return;
    
pauseTask:
    GST_WARNING("task paused with code %d\n", retVal);
    if (FALSE == gst_pad_pause_task(pad)){
        GST_ERROR("\n There is no task on this sink pad !! \n");
    }
    return;
    
}

/*=============================================================================
FUNCTION:            mfw_gst_rmdemuxer_fill_stream_info
        
DESCRIPTION:         Add and Creates the audio pad sets the 
					 corresponding capabilities.

ARGUMENTS PASSED:    demuxer      - The main structure of rm parser plugin context.
        
  
RETURN VALUE:
       TRUE       -	 if the function is performed properly
	   FALSE	  -	 if there is any errors encounterd
        

PRE-CONDITIONS:
   	    None
 
POST-CONDITIONS:
   	    None

IMPORTANT NOTES:
   	    None
=============================================================================*/

static gboolean
mfw_gst_rmdemuxer_fill_stream_info(MFW_GST_RMDEMUX_INFO_T * demuxer)
{

    GstPadTemplate *templ = NULL;
    GstCaps *caps = NULL;
    GstPad *pad = NULL;
    gchar *padname = NULL;
    guint total_pads;
    guint count;
    GstPad *src_pad;
    gboolean set;
    guint frame_width;
    guint frame_height;
	gchar *text_msg=NULL;
	guint sampling_frequency=0;
    gboolean foundstreaminfo = FALSE;

	//for multirate streams, we choose the maxrate streams to be played
	guint maxrarate =0;
	guint maxrvrate =0;
	int maxrastreamid = -1;
	int maxrvstreamid = -1;
	

    rm_stream_header * pHdr = HXNULL;
	rm_stream_header ** pHdrArray=HXNULL;
    rm_parser * rmparser = demuxer->rmparser;

    int i;
    UINT32 streams;
    streams = rm_parser_get_num_streams(rmparser);
    if (streams == 0)
    {
        GST_ERROR("can not find any stream !\n");
        return FALSE;
    }
    
    
    pHdrArray =(rm_stream_header **) RMMalloc(&pHdrArray,streams*sizeof(rm_stream_header *));
    if(pHdrArray==NULL)
    {
    	GST_ERROR("can not alloc enough memory !\n");
    	return FALSE;
    }

    RM_DEMUXER_FLOW("%d streams found. \n", streams);
    
    
    for (i=0;i<streams;i++)
    {
        pHdrArray[i]= NULL;
		    if (rm_parser_get_stream_header(rmparser, i, &pHdr)==HXR_OK)
		    {
		        pHdrArray[i] = pHdr;
		        if(rm_stream_is_realvideo(pHdr))
		        {
		            if(pHdr->ulAvgBitRate>maxrvrate)
		    	      {
		    	        maxrvstreamid = i;
				          maxrvrate = pHdr->ulAvgBitRate;
				        }
		        }
		       else if(rm_stream_is_realaudio(pHdr))
		       {
		          if(pHdr->ulAvgBitRate>maxrarate)
		    	    {
		    	      maxrastreamid = i;
				        maxrarate = pHdr->ulAvgBitRate;
		    	    }
		       }
		    }
		    else
		    {
            GST_ERROR("rm_parser_get_stream_header for %d failed\n");
		    }
    }
    
    if(rm_parser_is_multirate_file(rmparser))
    {
    	guint numofids=0;
    	guint selectedids[2];
    	
    	if(maxrastreamid!=-1)
    	{
    		if(maxrvstreamid!=-1)
    		{
    			numofids = 2;
    			selectedids[0] = maxrastreamid;
    			selectedids[1] = maxrvstreamid;
    		}
    		else{
    			numofids = 1;
    			selectedids[0] = maxrastreamid;
    		}
    	}
    	else if(maxrvstreamid!=-1)
    	{
    		 numofids = 1;
    		 selectedids[0] = maxrvstreamid;
    	}
    	g_print("nmumofids:%d\n",numofids);
    	if(numofids!=0)
    		rm_parser_select_streams(rmparser,selectedids,numofids);
    }
    
    
	  for(i=0;i<streams;i++)
	  {
	     if(pHdrArray[i]!=NULL && i!=maxrastreamid && i!=maxrvstreamid)
	  	     rm_parser_destroy_stream_header(rmparser, &pHdrArray[i]);
	  }	

    {//1
        caps = NULL;
        pHdr = NULL;
        rv_depack * rvdepack = NULL;
        rv_format_info * rvinfo = NULL;
        ra_depack * radepack = NULL;
        ra_format_info * rainfo = NULL;
        
        if (maxrvstreamid!=-1)
        {//2
                pHdr = pHdrArray[maxrvstreamid];
                if (demuxer->srcpad[VIDEO_INDEX]==NULL)
                {//3
                    if ((rvdepack 
                        = rv_depack_create2(demuxer,RMVideoFrameAvail, demuxer, 
                                                 RMError, RM_MEM_PRIVATE_OBJECT(demuxer), RMMalloc, RMFree))
                        ==NULL){
                        GST_ERROR("Failed in rv_depack_create2\n");
                        goto destroyStream;
                    }

                    if (rv_depack_init(rvdepack, pHdr)!=HXR_OK){
                        GST_ERROR("Failed in rv_depack_init\n");
                        goto destroyStream;
                    }

                    if (rv_depack_get_codec_init_info(rvdepack,&rvinfo)!=HXR_OK){
                        rvinfo = NULL;
                        GST_ERROR("Failed in rv_depack_get_codec_init_info\n");
                        goto destroyStream;
                    }

                    RM_DEMUXER_FLOW("Video found resolution:%dx%d fps:%d\n", 
                                    rvinfo->usWidth, rvinfo->usHeight,
                                    RM_FRAMERATE(rvinfo->ufFramesPerSecond));
                    
                    
                    demuxer->streamID[VIDEO_INDEX] = maxrvstreamid;

                    count = VIDEO_INDEX;

                    
                    char * media_type = "video/x-pn-realvideo";
                        padname = g_strdup_printf(PARSER_VIDEOPAD_TEMPLATE_NAME, maxrvstreamid);
                        templ = video_src_templ();
                        caps = gst_caps_new_simple(media_type,
                                    "height", G_TYPE_INT, rvinfo->usHeight,
                                    "width", G_TYPE_INT, rvinfo->usWidth, 
                                    "framerate", GST_TYPE_FRACTION,
                                    RM_FRAMERATE(rvinfo->ufFramesPerSecond), 
                                    1,
                                    NULL);

                    if (caps){//4

                        demuxer->streamMask |= VIDEO_MASK;
                        demuxer->caps[count] = caps;

                        pad = demuxer->srcpad[count] = gst_pad_new_from_template(templ, padname);

                        g_free(padname);

                        gst_pad_use_fixed_caps(pad);

                        /* set the stream_ptr on to the pad */
                        gst_pad_set_element_private(pad, demuxer->srcpad[count]);

                        /* set the capabilities of the pad */
                        set = gst_pad_set_caps(pad, caps);
                        if (set == FALSE) {
                            GST_ERROR
                                ("\n unable to set the capability of the pad:result is %d\n", set);
                            goto destroyStream;
                        }

                        /* check if the caps represents media formats */
                        GstCaps *temp_caps;
                        temp_caps = gst_pad_get_caps(pad);
                        if (gst_caps_is_empty(temp_caps)){
                            GST_ERROR("\n caps is empty\n");
                            gst_caps_unref(temp_caps);
                            gst_caps_unref(caps);
                                
                            goto destroyStream;
                        }
                        
                        gst_caps_unref(temp_caps);
                        /* Activates the given pad */
                        if (gst_pad_set_active(pad, TRUE) == TRUE) {
                        	  g_print("set active video pad successful!\n");
                            GST_DEBUG("\nThe pad was activated successfully\n");
                        }
                        /* adds a pad to the element */
                        gst_element_add_pad(GST_ELEMENT(demuxer), pad);

                        GST_LOG_OBJECT(GST_ELEMENT(demuxer), "Added pad %s with caps %p",
                        	        GST_PAD_NAME(pad), caps);
                        /* unref a GstCap */
                        gst_caps_unref(caps);

                        demuxer->rvdepack = rvdepack;
                        demuxer->rvinfo = rvinfo;
                        demuxer->rvheader = pHdr;
                        
                        foundstreaminfo = TRUE;
                        //continue;
                    }   
             }
      }
				
			if (maxrastreamid!=-1){
                pHdr = pHdrArray[maxrastreamid];
                if (demuxer->srcpad[AUDIO_INDEX]==NULL){
                    UINT32 substreams, fourcc;
                    
                    if ((radepack 
                        = ra_depack_create2(demuxer,RMAudioFrameAvail, demuxer, 
                                                 RMError, RM_MEM_PRIVATE_OBJECT(demuxer), RMMalloc, RMFree))
                        ==NULL){
                        GST_ERROR("Failed in ra_depack_create2\n");
                        goto destroyStream;
                    }

                    if (ra_depack_init(radepack, pHdr)!=HXR_OK){
                        GST_ERROR("Failed in ra_depack_init\n");
                        goto destroyStream;
                    }

                    substreams = ra_depack_get_num_substreams(radepack);

                    fourcc = ra_depack_get_codec_4cc(radepack,0);

                    if (ra_depack_get_codec_init_info(radepack, 0, &rainfo)!=HXR_OK){
                        rainfo = NULL;
                        GST_ERROR("Failed in ra_depack_get_codec_init_info\n");
                        goto destroyStream;
                    }

                    demuxer->streamID[AUDIO_INDEX] = maxrastreamid;
                    count = AUDIO_INDEX;

                    switch(fourcc){
                        case RM_FOURCC_RA8LOW:
                        case RM_FOURCC_RAVOICE:
                            if ((rainfo->usFlavorIndex>29)||(rainfo->usNumChannels>2)){
                                RM_DEMUXER_FLOW("Audio(fourcc:%c%c%c%c) Flavorindex:%d, channel:%d not support!\n", 
                                    FOURCCTO4CHAR(fourcc), rainfo->usFlavorIndex, rainfo->usNumChannels);
                                goto destroyStream;
                            }
                            RM_DEMUXER_FLOW("Audio (fourcc:%c%c%c%c) found channel:%d samplerate:%ld\n", 
                                            FOURCCTO4CHAR(fourcc),
                                            rainfo->usNumChannels, rainfo->ulSampleRate);
                            char * media_type="audio/x-pn-realaudio";
                            padname = g_strdup_printf(PARSER_AUDIOPAD_TEMPLATE_NAME, maxrastreamid);
                            templ = audio_src_templ();
                            caps = gst_caps_new_simple(media_type,
                                        "flavorindex", G_TYPE_INT, rainfo->usFlavorIndex,
                                        "bitrate", G_TYPE_INT,
                                        rainfo->ulActualRate,
                                        "rate", G_TYPE_INT,
                                        rainfo->ulSampleRate,
                                        "channels",G_TYPE_INT,
                                        rainfo->usNumChannels,
                                        NULL);
                            break;
                        case RM_FOURCC_AAC:
                            padname = g_strdup_printf(PARSER_AUDIOPAD_TEMPLATE_NAME, maxrastreamid);
                            templ = audio_src_templ();
                            media_type="audio/mpeg";
                            caps = gst_caps_new_simple(media_type,
                                        "mpegversion", G_TYPE_INT, 4,
                                        "bitrate", G_TYPE_INT,
                                        rainfo->ulActualRate,
                                        "rate", G_TYPE_INT,
                                        rainfo->ulSampleRate,
                                        "channels",G_TYPE_INT,
                                        rainfo->usNumChannels,
                                        NULL);
                            break;
                        default:
                            RM_DEMUXER_FLOW("Audio (fourcc:%c%c%c%c) not support, flavorindex:%d!\n", FOURCCTO4CHAR(fourcc), rainfo->usFlavorIndex);
                            goto destroyStream;
                            break;
                    }

                     if (caps){

                        demuxer->streamMask |= AUDIO_MASK;
                        demuxer->caps[count] = caps;

                        pad = demuxer->srcpad[count] = gst_pad_new_from_template(templ, padname);

                        g_free(padname);

                        gst_pad_use_fixed_caps(pad);

                        /* set the stream_ptr on to the pad */
                        gst_pad_set_element_private(pad, demuxer->srcpad[count]);

                        /* set the capabilities of the pad */
                        set = gst_pad_set_caps(pad, caps);
                        if (set == FALSE) {
                            GST_ERROR
                                ("\n unable to set the capability of the pad:result is %d\n", set);
                            goto destroyStream;
                        }

                        GstCaps *temp_caps;
                        temp_caps = gst_pad_get_caps(pad);
                        if (gst_caps_is_empty(temp_caps)){
                            GST_ERROR("\n caps is empty\n");
                            gst_caps_unref(temp_caps);
                                gst_caps_unref(caps);
                            goto destroyStream;
                        }
                        gst_caps_unref(temp_caps);

                        /* Activates the given pad */
                        if (gst_pad_set_active(pad, TRUE) == TRUE) {
                            GST_DEBUG("\nThe pad was activated successfully\n");
                        }
                        /* adds a pad to the element */
                        gst_element_add_pad(GST_ELEMENT(demuxer), pad);

                        GST_LOG_OBJECT(GST_ELEMENT(demuxer), "Added pad %s with caps %p",
                        	        GST_PAD_NAME(pad), caps);
                        /* unref a GstCap */
                        gst_caps_unref(caps);

                        demuxer->radepack = radepack;
                        demuxer->rainfo = rainfo;
                        demuxer->raheader = pHdr;
                    
                        foundstreaminfo = TRUE;
                        //continue;
                    }
               }
          }
          
         goto goodreturn;
          
destroyStream:
            if (rvinfo){
                rv_depack_destroy_codec_init_info(rvdepack, &rvinfo);
            }
            if (rvdepack){
                rv_depack_destroy(&rvdepack);
            }
            if (rainfo){
                ra_depack_destroy_codec_init_info(radepack, &rainfo);
            }
            if (radepack){
                ra_depack_destroy(&radepack);
            }
            if (maxrastreamid!=-1){
            	if(pHdrArray[maxrastreamid])
            		rm_parser_destroy_stream_header(rmparser, &pHdrArray[maxrastreamid]);
            }
            if (maxrvstreamid!=-1){
            	if(pHdrArray[maxrvstreamid])
            		rm_parser_destroy_stream_header(rmparser, &pHdrArray[maxrvstreamid]);
            }
    }
goodreturn:
	
    if(pHdrArray)
    RMFree(pHdrArray,pHdrArray);
    
    return foundstreaminfo;
}

/*=============================================================================
FUNCTION:            rm_demuxer_set_pad_caps
        
DESCRIPTION:         Reset the capabilities of the pads created before.

ARGUMENTS PASSED:    demuxer      - The main structure of rm parser plugin context.
        
  
RETURN VALUE:
       TRUE       -	 if the function is performed properly
	   FALSE	  -	 if there is any errors encounterd
        

PRE-CONDITIONS:
   	    None
 
POST-CONDITIONS:
   	    None

IMPORTANT NOTES:
   	    None
   	
=============================================================================*/

static gboolean rm_demuxer_set_pad_caps(MFW_GST_RMDEMUX_INFO_T * demuxer)
{
    gboolean set;

    if (demuxer->srcpad[AUDIO_INDEX]){
        set = gst_pad_set_caps(demuxer->srcpad[AUDIO_INDEX],demuxer->caps[AUDIO_INDEX]);
        if (set == FALSE) {
            GST_ERROR("\n unable to set the capability of the pad:result is %d\n",
                 set);
            return FALSE;
	    }
    }

    if (demuxer->srcpad[VIDEO_INDEX]){
        set = gst_pad_set_caps(demuxer->srcpad[VIDEO_INDEX],demuxer->caps[VIDEO_INDEX]);
        if (set == FALSE) {
            GST_ERROR("\n unable to set the capability of the pad:result is %d\n",
                 set);
            return FALSE;
	    }
    }
    return TRUE;
}


/*=============================================================================
FUNCTION:            audio_src_templ
        
DESCRIPTION:         defines the audio source pad template.

ARGUMENTS PASSED:    None
  
RETURN VALUE:        Pointer to GstPadTemplate. 

PRE-CONDITIONS:      None
   	    
POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/
static GstPadTemplate *audio_src_templ(void)
{
    static GstPadTemplate *templ = NULL;

    if (!templ) {
	GstCaps *caps = NULL, * othercaps;
	GstStructure *structure;

	caps = gst_caps_new_simple("audio/x-pn-realaudio",
				   NULL);

    othercaps = gst_caps_new_simple("audio/mpeg",
				   NULL);

    gst_caps_append(caps,othercaps);

	templ = gst_pad_template_new(PARSER_AUDIOPAD_TEMPLATE_NAME, GST_PAD_SRC,
				     GST_PAD_SOMETIMES, caps);
    }
    return templ;
}

/*=============================================================================
FUNCTION:            video_src_templ
        
DESCRIPTION:         defines the video source pad template.

ARGUMENTS PASSED:    None
  
RETURN VALUE:        Pointer to GstPadTemplate. 

PRE-CONDITIONS:      None
 
POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/
static GstPadTemplate *video_src_templ()
{
    static GstPadTemplate *templ = NULL;

    if (!templ) {
	GstCaps *caps      = NULL;
	GstStructure *structure;

	caps     = gst_caps_new_simple(MFW_GST_RM_VIDEO_CAPS,
				   NULL
				   );
				   
	templ = gst_pad_template_new(PARSER_VIDEOPAD_TEMPLATE_NAME, GST_PAD_SRC,
				     GST_PAD_SOMETIMES, caps);
    }
    return templ;
}



/*=============================================================================
FUNCTION:            mfw_gst_rmdemuxer_sink_event
        
DESCRIPTION:         This functions handles the events that triggers the
				     sink pad of the rmdemuxer element.

ARGUMENTS PASSED:
        pad      -   pointer to pad
        event    -   pointer to event
        
RETURN VALUE:
       TRUE       -	 if event is sent to sink properly
	   FALSE	  -	 if event is not sent to sink properly

PRE-CONDITIONS:
   	    None
 
POST-CONDITIONS:
   	    None

IMPORTANT NOTES:
   	    None
=============================================================================*/
static gboolean mfw_gst_rmdemuxer_sink_event(GstPad * pad,
					      GstEvent * event)
{
    gboolean result = TRUE;
    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_NEWSEGMENT:
	{
	    GstFormat format;
	    gst_event_parse_new_segment(event, NULL, NULL, &format, NULL,
					NULL, NULL);
	    if (format == GST_FORMAT_TIME) {
		GST_ERROR("\nCame to the FORMAT_TIME call\n");
		gst_event_unref(event);
		result = TRUE;
	    } else {
		GST_ERROR("dropping newsegment event in format %s",
			gst_format_get_name(format));
		gst_event_unref(event);
		result = TRUE;
	    }
	    break;
	}

    case GST_EVENT_EOS:
	{
	    RM_DEMUXER_FLOW("\nDemuxer: Sending EOS to Decoder\n", 0);
	    result = gst_pad_push_event(pad, event);

	    if (result != TRUE) {
		GST_ERROR("\n Error in pushing the event, result is %d\n",
			  result);
	    }
	    break;
	}

    default:
	{
	    result = gst_pad_event_default(pad, event);
	    return TRUE;
	}

    }

    return result;
}

/*=============================================================================
FUNCTION:   	     mfw_gst_rmdemuxer_activate

DESCRIPTION:         it will call gst_pad_activate_pull which activates or
                     deactivates the given pad in pull mode via dispatching to the
                     pad's activepullfunc

ARGUMENTS PASSED:
         pad       - pointer to GstPad to activate or deactivate

RETURN VALUE:
         TRUE      - if operation was successful
         FALSE     - if operation was unsuccessful

PRE-CONDITIONS:      None
        

POST-CONDITIONS:     None
        

IMPORTANT NOTES:     None
        
=============================================================================*/

static gboolean mfw_gst_rmdemuxer_activate(GstPad * pad)
{


    if (gst_pad_check_pull_range(pad)) {
	return gst_pad_activate_pull(pad, TRUE);
    } else {
	return FALSE;
    }

    return TRUE;

}

/*=============================================================================
FUNCTION:			 mfw_gst_rmdemuxer_activate_pull

DESCRIPTION:		 This will start the demuxer task function in "pull" based	
					 scheduling model.

ARGUMENTS PASSED:
	pad		         sink pad, where the data will be pulled into, from	
					 file source.
	active 	         pad is active or not(TRUE/FALSE)
 

RETURN VALUE:		
		TRUE      -  if operation was successful
        FALSE     -  if operation was unsuccessful

PRE-CONDITIONS:      None

POST-CONDITIONS:     None

IMPORTANT NOTES:     None
=============================================================================*/

static gboolean
mfw_gst_rmdemuxer_activate_pull(GstPad * pad, gboolean active)
{
    gboolean ret_val = TRUE;
    MFW_GST_RMDEMUX_INFO_T *demuxer =
	    MFW_GST_RM_DEMUXER(GST_PAD_PARENT(pad));
    
    if (active){

        /* start a Task, which will read each frame of audio/video data from the 
        container */
        ret_val = gst_pad_start_task(demuxer->sinkpad,
                                     (GstTaskFunction)mfw_gst_rmdemuxer_taskfunc,
                                     demuxer->sinkpad);

        if (ret_val == FALSE) {
            GST_ERROR("Task could not be started \n");
            mfw_gst_rm_demuxer_close(demuxer);
            return ret_val;
        }
    }else{
	    /* pause the Task */
	    return gst_pad_pause_task(demuxer->sinkpad);

    }

    return TRUE;
}


/*=============================================================================
FUNCTION:   	     mfw_gst_rmdemuxer_change_state
	
DESCRIPTION:	     This function keeps track of different states of pipeline.

ARGUMENTS PASSED:
      element    :   pointer to the rm demuxer element.
	  transition :   state of the pipeline.
     
RETURN VALUE       
        GST_STATE_CHANGE_FAILURE    - the state change failed
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded
        GST_STATE_CHANGE_ASYNC      - the state change will happen
                                        asynchronously
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled
        

PRE-CONDITIONS:      None
        

POST-CONDITIONS:     None
        

IMPORTANT NOTES:     None
        
=============================================================================*/

static GstStateChangeReturn
mfw_gst_rmdemuxer_change_state(GstElement * element,
				GstStateChange transition)
{
    GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
    MFW_GST_RMDEMUX_INFO_T *rm_demuxer = MFW_GST_RM_DEMUXER(element);
    guint32 stream_count;

    GstFormat fmt = GST_FORMAT_TIME;
    gint64 pos = 0, len = 0;
    gboolean ret_value;
    guint count;

	/***** UPWARDS *****/
    switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
	{
	    GST_DEBUG
		("\nCame to Demuxer GST_STATE_CHANGE_NULL_TO_READY\n");

	    rm_demuxer->file_info.length = 0;
	    rm_demuxer->file_info.offset = 0;
	    rm_demuxer->segment.last_stop = 0;
	    rm_demuxer->new_seg_flag = 0;
	    rm_demuxer->stop_request = FALSE;
		rm_demuxer->seek_flag = FALSE;
        rm_demuxer->streamMask = 0;
        rm_demuxer->eosfile = FALSE;
        rm_demuxer->rmparser = NULL;
        rm_demuxer->radepack = NULL;
        rm_demuxer->rvdepack = NULL;
        rm_demuxer->rvinfo = NULL;
        rm_demuxer->rainfo = NULL;
        rm_demuxer->raheader = NULL;
        rm_demuxer->rvheader = NULL;
        rm_demuxer->firstaudio = TRUE;
        rm_demuxer->streamID[AUDIO_INDEX] = -1;
        rm_demuxer->streamID[VIDEO_INDEX] = -1;
        rm_demuxer->vhead = rm_demuxer->vtail = NULL;
        rm_demuxer->vframenum = 0;
        rm_demuxer->new_seg_time = 0;
        rm_demuxer->actual_seek_time = 0;
        SYNCER_INIT(rm_demuxer->audiosync);
        SYNCER_INIT(rm_demuxer->videosync);
        rm_demuxer->videokeyfound = FALSE;
        rm_demuxer->videodropto = FALSE;
        rm_demuxer->audiokeyfound = FALSE;
        rm_demuxer->videosent = FALSE;
        rm_demuxer->readerror = FALSE;
        rm_demuxer->freshstart = TRUE;
        rm_demuxer->seek_video = TRUE;
        
#ifdef MEMORY_DEBUG
        init_memmanager(&rm_demuxer->memmgr, "RM");
#endif
	    break;
	}

    case GST_STATE_CHANGE_READY_TO_PAUSED:
	{

	    GST_DEBUG
		("\nCame to Demuxer GST_STATE_CHANGE_READY_TO_PAUSED\n");
       
        /* Registers a new tag type for the use with GStreamer's type system */
	    gst_tag_register (MFW_GST_TAG_WIDTH, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "image width","image width(pixel)", NULL);
        gst_tag_register (MFW_GST_TAG_HEIGHT, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "image height","image height(pixel)", NULL); 
	    gst_tag_register (MFW_GST_TAG_FRAMERATE, GST_TAG_FLAG_DECODED,G_TYPE_FLOAT,
            "video framerate","number of video frames in a second", NULL);
	    gst_tag_register (MFW_GST_TAG_SAMPLING_FREQUENCY, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "sampling frequency","number of audio samples per frame per second", NULL);
	    gst_tag_register (MFW_GST_TAG_YEAR, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
            "year","year of creation", NULL);

	    /* Initialize segment to its default values. */

	    gst_segment_init(&rm_demuxer->segment, GST_FORMAT_UNDEFINED);

	    /* Sets the given activate function for the pad. The activate function will
	       dispatch to activate_push or activate_pull to perform the actual activation. */

	    gst_pad_set_activate_function(rm_demuxer->sinkpad,
					  mfw_gst_rmdemuxer_activate);

	    /* Sets the given activate_pull function for the pad. An activate_pull
	       function prepares the element and any upstream connections for pulling. */

	    gst_pad_set_activatepull_function(rm_demuxer->sinkpad,
					      GST_DEBUG_FUNCPTR
					      (mfw_gst_rmdemuxer_activate_pull));
        break;
	}

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
	{

	    GST_DEBUG
		("\nCame to Demuxer GST_STATE_CHANGE_PAUSED_TO_PLAYING\n");
	    break;
	}

    default:
	break;

    }

    
    ret = parent_class->change_state(element, transition);
	


	 /***** DOWNWARDS *****/
    switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
	{
	    break;
	}
    case GST_STATE_CHANGE_PAUSED_TO_READY:
	{
		/* received stop request,do the following */

        mfw_gst_rm_demuxer_close(rm_demuxer);
	   
	    /* Initialize segment to its default values. */
	    gst_segment_init(&rm_demuxer->segment, GST_FORMAT_UNDEFINED);

	    rm_demuxer->file_info.length = 0;
	    rm_demuxer->file_info.offset = 0;
	    rm_demuxer->segment.last_stop = 0;
	    rm_demuxer->new_seg_flag = 0;
	    rm_demuxer->stop_request = TRUE;
		rm_demuxer->seek_flag=FALSE;
       
	    break;

	}

    case GST_STATE_CHANGE_READY_TO_NULL:
	{
#ifdef MEMORY_DEBUG
        deinit_memmanager(&rm_demuxer->memmgr);
#endif 
	    break;
	}

    default:
	break;
    }

    return ret;
}




/*=============================================================================
FUNCTION:   mfw_gst_rmdemuxer_handle_sink_query   

DESCRIPTION:    performs query on src pad.    

ARGUMENTS PASSED:
        pad     -   pointer to GstPad
        query   -   pointer to GstQuery        
            
RETURN VALUE:
        TRUE    -   success
        FALSE   -   failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/

static const GstQueryType *gst_rmdemuxer_get_src_query_types(GstPad * pad)
{
    static const GstQueryType types[] = {
	//GST_QUERY_POSITION,
	GST_QUERY_DURATION,
	0
    };

    return types;
}

/*=============================================================================
FUNCTION:   mfw_gst_rmdemuxer_handle_sink_query   

DESCRIPTION:    performs query on src pad.    

ARGUMENTS PASSED:
        pad     -   pointer to GstPad
        query   -   pointer to GstQuery        
            
RETURN VALUE:
        TRUE    -   success
        FALSE   -   failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/

static gboolean mfw_gst_rmdemuxer_handle_src_query(GstPad * pad,
						    GstQuery * query)
{
    MFW_GST_RMDEMUX_INFO_T *demuxer_info;
    gboolean res = TRUE;
    demuxer_info = MFW_GST_RM_DEMUXER(gst_pad_get_parent(pad));

    GstFormat format;
    format = GST_FORMAT_TIME;
    gint64 dest_value;

    //RM_DEMUXER_FLOW("handling %s query\n",
	//      gst_query_type_get_name(GST_QUERY_TYPE(query)));

    switch (GST_QUERY_TYPE(query)) {
    case GST_QUERY_DURATION:
	{
		
	    /* save requested format */
	    gst_query_parse_duration(query, &format, NULL);
	  	    

	    if (format == GST_FORMAT_TIME) {

		gst_query_set_duration(query, GST_FORMAT_TIME,
				       demuxer_info->segment.duration);
	    } else if (format == GST_FORMAT_BYTES) {
		gst_query_set_duration(query, GST_FORMAT_BYTES,
				       demuxer_info->segment.duration);

	    }

	    res = TRUE;
	    break;
	}
#if 0    
    case GST_QUERY_POSITION:
	{
	    GstFormat format;

	    /* save requested format */
	    gst_query_parse_position(query, &format, NULL);

	    gst_query_set_position(query, GST_FORMAT_TIME,
				   demuxer_info->segment.last_stop);

	    res = TRUE;
	    break;
	}
#endif    
    default:
	res = FALSE;

    }
    gst_object_unref(demuxer_info);
    return res;

}



/*=============================================================================
FUNCTION:   	mfw_gst_rmdemuxer_handle_src_event

DESCRIPTION:	Handles an event on the source pad.

ARGUMENTS PASSED:
        pad        -    pointer to pad
        event      -    pointer to event
RETURN VALUE:
        TRUE       -	if event is sent to source properly
	    FALSE	   -	if event is not sent to source properly

PRE-CONDITIONS:    None

POST-CONDITIONS:   None

IMPORTANT NOTES:   None
=============================================================================*/

gboolean mfw_gst_rmdemuxer_handle_src_event(GstPad* src_pad, GstEvent* event)
{
    gboolean res = TRUE;

	MFW_GST_RMDEMUX_INFO_T *demuxer_info = MFW_GST_RM_DEMUXER(GST_PAD_PARENT (src_pad));
 
	switch (GST_EVENT_TYPE (event)) 
	{
		case GST_EVENT_SEEK:
		{
			GstSeekFlags flags;
	  	    gboolean     flush;
			gdouble      rate;
			GstSeekType  cur_type;
	        GstSeekType	 stop_type;
	        gint64       cur;
	        GstFormat    format;
	        gint64       stop;
           
            GST_DEBUG("Handling the seek event\n");

			gst_event_ref (event);

			/* parse a seek event */
            gst_event_parse_seek(event, &rate, &format, &flags,
			&cur_type, &cur, &stop_type, &stop);
            /* demuxer cannot do seek if format is not bytes */
            if ((format != GST_FORMAT_TIME))
            {
                GST_WARNING("SEEK event not TIME format.\n");
                gst_event_unref (event);
                return TRUE;
 
            }

			/* if the seek time is greater than the stream duration */
            if(cur > demuxer_info->segment.duration) {
                GST_WARNING("SEEK event exceed the maximum duration.\n");
                cur = demuxer_info->segment.duration;
            }   

            RM_DEMUXER_FLOW("seeking to %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(cur));
            
            flush = flags & GST_SEEK_FLAG_FLUSH;
		    if(flush)
            {
                guint strem_count;
				/* sends an event to upstream elements with event as new flush start event*/
                res = gst_pad_push_event (demuxer_info->sinkpad, gst_event_new_flush_start ());
                if (!res) 
				{
                    GST_WARNING_OBJECT (demuxer_info, "Failed to push event upstream!");
                }
                for(strem_count = 0; strem_count < MAX_SRC_PADS; strem_count++)
                {
                    if (demuxer_info->srcpad[strem_count]){
                        /* sends an event to downstream elements with event as new flush start event*/
                        
                        res = gst_pad_push_event (demuxer_info->srcpad[strem_count], 
    						                      gst_event_new_flush_start());
    						        
                        if (!res)
    					{
                            GST_WARNING_OBJECT (demuxer_info, "Failed to push event downstream!");
                        }
                    }
                }
            }

            gst_pad_pause_task (demuxer_info->sinkpad);
				  
		  	
		
			/*Lock the stream lock of sinkpad*/
			GST_PAD_STREAM_LOCK(demuxer_info->sinkpad);
		
			/* perform seek */

            demuxer_info->desired_seek_time = cur;

            demuxer_info->audiokeyfound = FALSE;

            demuxer_info->videokeyfound = FALSE;
			demuxer_info->new_seg_flag = (demuxer_info->streamMask|SEGMENT_NOT_SEND_MASK);
            
			/* sends an event to upstream elements with event as new flush stop event
			   , to resume data flow */
			res = gst_pad_push_event (demuxer_info->sinkpad, gst_event_new_flush_stop ());
   	        
            if(res == FALSE)
			{
              GST_WARNING_OBJECT (demuxer_info, "Failed to push event upstream!");
			}
			
            
			 /* send flush stop to down stream elements,to enable data flow*/
            if(flush)
			{
              guint strem_count;
              for(strem_count = 0; strem_count < MAX_SRC_PADS; strem_count++)
              {
                   if (demuxer_info->srcpad[strem_count]){       
                      res = gst_pad_push_event (demuxer_info->srcpad[strem_count],
    					                      gst_event_new_flush_stop());
                      if (!res) 
    				  {
                          GST_WARNING_OBJECT (demuxer_info, "Failed to push event downstream!");
                      }
                   }
              }
          }
         
		 /*Unlock the stream lock of sink pad*/	
         GST_PAD_STREAM_UNLOCK(demuxer_info->sinkpad);
         demuxer_info->seek_flag=TRUE;
         demuxer_info->seek_video=TRUE;

          /* streaming can continue now */
          gst_pad_start_task (demuxer_info->sinkpad,
              (GstTaskFunction)mfw_gst_rmdemuxer_taskfunc,demuxer_info->sinkpad);

          break;
		}

		default:
		res = FALSE;
		break;
	}
		
	gst_event_unref (event);
	return TRUE;
}





/*=============================================================================
FUNCTION:            mfw_gst_rm_demuxer_close
        
DESCRIPTION:         This function  Frees All Memories Allocated and Resourses. 

ARGUMENTS PASSED:    None
        
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

void mfw_gst_rm_demuxer_close(MFW_GST_RMDEMUX_INFO_T *rm_demuxer)
{

    mfw_gst_clean_all_queue(rm_demuxer);
    mfw_gst_rmdemuxer_core_cleanup(rm_demuxer);
    return;

}


/*=============================================================================
FUNCTION:            mfw_gst_type_rm_demuxer_get_type
        
DESCRIPTION:         intefaces are initiated in this function.you can register one
                     or more interfaces  after having registered the type itself.


ARGUMENTS PASSED:    None
        
  
RETURN VALUE:        A numerical value ,which represents the unique identifier of this
                     element(rmdemuxer)

        

PRE-CONDITIONS:      None
   	 
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

GType mfw_gst_type_rm_demuxer_get_type(void)
{
    static GType rmdemuxer_type = 0;

    if (!rmdemuxer_type) {
	static const GTypeInfo rmdemuxer_info = {
	    sizeof(MFW_GST_RMDEMUX_INFO_CLASS_T),
	    (GBaseInitFunc) mfw_gst_rm_demuxer_base_init,
	    NULL,
	    (GClassInitFunc) mfw_gst_rm_demuxer_class_init,
	    NULL,
	    NULL,
	    sizeof(MFW_GST_RMDEMUX_INFO_T),
	    0,
	    (GInstanceInitFunc) mfw_gst_rm_demuxer_init,
	};
	rmdemuxer_type = g_type_register_static(GST_TYPE_ELEMENT,
						 "MFW_GST_RMDEMUX_INFO_T",
						 &rmdemuxer_info, 0);

	GST_DEBUG_CATEGORY_INIT(mfw_gst_rmdemuxer_debug, "mfw_rmdemuxer",
				0, "rm demuxer");
    }
    return rmdemuxer_type;
}

/*=============================================================================
FUNCTION:            mfw_gst_rm_demuxer_base_init    
        
DESCRIPTION:         rmdemuxer element details are registered with the plugin during
                     _base_init ,This function will initialise the class and child
                     class properties during each new child class creation

ARGUMENTS PASSED:
        Klass   -    pointer to rmdemuxer plug-in class
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

static void
mfw_gst_rm_demuxer_base_init(MFW_GST_RMDEMUX_INFO_CLASS_T * klass)
{

    GST_DEBUG("Registering the element details with the plugin\n");

    static GstElementDetails mfw_gst_rmdemuxer_details = {
	"Freescale RealMedia demuxer",
	"Codec/Demuxer",
	"Demultiplex an rm data into audio and video data",
	FSL_GST_MM_PLUGIN_AUTHOR
    };

    GstElementClass *element_class = GST_ELEMENT_CLASS(klass);
    gst_element_class_add_pad_template(element_class, audio_src_templ());
	gst_element_class_add_pad_template(element_class, video_src_templ());

    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&mfw_gst_rm_demuxer_sink_factory));

    gst_element_class_set_details(element_class,
				  &mfw_gst_rmdemuxer_details);

    return;
}

/*=============================================================================
FUNCTION:            mfw_gst_rm_demuxer_class_init
        
DESCRIPTION:         Initialise the class only once (specifying what signals,
                     arguments and virtual functions the class has and setting up
                     global state)

ARGUMENTS PASSED:    klass   - pointer to rmdemuxer element class
        
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

static void
mfw_gst_rm_demuxer_class_init(MFW_GST_RMDEMUX_INFO_CLASS_T * klass)
{
    GObjectClass *gobject_class = NULL;
    GstElementClass *gstelement_class = NULL;

    GST_DEBUG("Initialise the demuxer class\n");

    gobject_class = (GObjectClass *) klass;
    gstelement_class = (GstElementClass *) klass;
    parent_class = g_type_class_peek_parent(klass);

    gstelement_class->change_state = mfw_gst_rmdemuxer_change_state;

    return;
}

/*=============================================================================
FUNCTION:            mfw_gst_rm_demuxer_init
        
DESCRIPTION:         This function creates the pads on the elements and register the
			         function pointers which operate on these pads. 

ARGUMENTS PASSED:    
      demuxer_info  :pointer to the rmdemuxer plugin structure.
	  gclass        :pointer to rmdemuxer element class.
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None 
   	  
 
POST-CONDITIONS:     None
 

IMPORTANT NOTES:     None
   	   
=============================================================================*/
static void
mfw_gst_rm_demuxer_init(MFW_GST_RMDEMUX_INFO_T * demuxer_info,
			 MFW_GST_RMDEMUX_INFO_CLASS_T * gclass)
{

    gchar *padname = NULL;
    GstCaps *caps = NULL;
    GstPadTemplate *templ = NULL;
    GstPad *pad = NULL;
    gboolean set;


    GstElementClass *klass = GST_ELEMENT_GET_CLASS(demuxer_info);
    
    demuxer_info->sinkpad =
	gst_pad_new_from_template(gst_element_class_get_pad_template
				  (klass, "sink"), "sink");

    GST_DEBUG("Register the function pointers on to the sink pad\n");

    gst_pad_set_setcaps_function(demuxer_info->sinkpad,
				 mfw_gst_rm_demuxer_set_caps);

    gst_element_add_pad(GST_ELEMENT(demuxer_info), demuxer_info->sinkpad);

    gst_pad_set_event_function(demuxer_info->sinkpad,
			       GST_DEBUG_FUNCPTR
			       (mfw_gst_rmdemuxer_sink_event));

#define MFW_GST_RM_DEMUXER_PLUGIN VERSION
    PRINT_CORE_VERSION(FSL_RM_Parser_Version_Info());
    PRINT_PLUGIN_VERSION(MFW_GST_RM_DEMUXER_PLUGIN);  

    return;
}

/*=============================================================================
FUNCTION:            mfw_gst_rm_demuxer_set_caps
         
DESCRIPTION:         this function handles the link with other plug-ins and used for
                     capability negotiation  between pads  

ARGUMENTS PASSED:    
        pad      -   pointer to GstPad
        caps     -   pointer to GstCaps
        
  
RETURN VALUE:        None
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	    

IMPORTANT NOTES:     None
   	    
=============================================================================*/

static gboolean mfw_gst_rm_demuxer_set_caps(GstPad * pad, GstCaps * caps)
{

    MFW_GST_RMDEMUX_INFO_T *demuxer_info;
    GstStructure *structure = gst_caps_get_structure(caps, 0);
    demuxer_info = MFW_GST_RM_DEMUXER(gst_pad_get_parent(pad));
    gst_object_unref(demuxer_info);
    return TRUE;
}

/*=============================================================================
FUNCTION:            plugin_init
        
ESCRIPTION:          special function , which is called as soon as the plugin or
                     element is loaded and information returned by this function
                     will be cached in central registry


ARGUMENTS PASSED:    
          plugin -   pointer to container that contains features loaded
                     from shared object module
        
  
RETURN VALUE:        return TRUE or FALSE depending on whether it loaded initialized any
                     dependency correctly
        

PRE-CONDITIONS:      None
   	    
 
POST-CONDITIONS:     None
   	   

IMPORTANT NOTES:     None
   	    
=============================================================================*/


static gboolean plugin_init(GstPlugin * plugin)
{
    return gst_element_register(plugin, "mfw_rmdemuxer",
				FSL_GST_RANK_HIGH,
				MFW_GST_TYPE_RM_DEMUXER);
}

/*   This is used to define the entry point and meta data of plugin   */

GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,	/* major version of Gstreamer  */
		  GST_VERSION_MINOR,	/* minor version of Gstreamer  */
		  "mfw_rmdemuxer",	/* name of the plugin          */
		  "demuxes video and audio streams from realmedia file",	/* what plugin actually does   */
		  plugin_init,	/* first function to be called */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  FSL_GST_MM_PLUGIN_PACKAGE_NAME, FSL_GST_MM_PLUGIN_PACKAGE_ORIG)

